<?php

namespace Botble\BbFormBuilder\Tests;

use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Providers\FormBuilderServiceProvider;
use Botble\LanguageAdvanced\Supports\LanguageAdvancedManager;
use Tests\TestCase;

class FormBuilderPhase02Test extends TestCase
{
    public function testFormBuilderServiceProviderHasValidPhpSyntax(): void
    {
        $filePath = base_path('platform/plugins/bb-form-builder/src/Providers/FormBuilderServiceProvider.php');
        $this->assertFileExists($filePath);

        $output = null;
        $returnCode = null;
        exec('php -l ' . escapeshellarg($filePath), $output, $returnCode);

        $this->assertEquals(0, $returnCode, 'FormBuilderServiceProvider has PHP syntax errors: ' . implode('\n', $output ?? []));
    }

    public function testLanguageAdvancedManagerSupportsFormBuilderModel(): void
    {
        if (! is_plugin_active('language-advanced')) {
            $this->markTestSkipped('language-advanced plugin is not active');
        }

        $this->assertTrue(
            LanguageAdvancedManager::isSupported(FormBuilder::class),
            'FormBuilder model should be supported by LanguageAdvancedManager'
        );
    }

    public function testLanguageAdvancedManagerReturnsCorrectTranslatableColumns(): void
    {
        if (! is_plugin_active('language-advanced')) {
            $this->markTestSkipped('language-advanced plugin is not active');
        }

        $columns = LanguageAdvancedManager::getTranslatableColumns(FormBuilder::class);

        $this->assertIsArray($columns);
        $this->assertContains('name', $columns, 'name column should be translatable');
        $this->assertContains('content', $columns, 'content column should be translatable');
        $this->assertContains('properties', $columns, 'properties column should be translatable');
        $this->assertContains('submission', $columns, 'submission column should be translatable');

        $expectedColumns = ['name', 'content', 'properties', 'submission'];
        $this->assertEquals($expectedColumns, $columns, 'Translatable columns should match exactly');
    }

    public function testFormBuilderModelHasTranslationsRelationship(): void
    {
        if (! is_plugin_active('language-advanced')) {
            $this->markTestSkipped('language-advanced plugin is not active');
        }

        $form = FormBuilder::query()->firstOrCreate(
            ['code' => 'test-form-phase02-' . time()],
            [
                'name' => 'Test Form Phase 02',
                'status' => 'published',
            ]
        );

        try {
            $translations = $form->translations;
            $this->assertIsIterable($translations, 'translations relationship should be iterable');
            $this->assertTrue(true, 'FormBuilder model has working translations relationship');
        } catch (\Exception $e) {
            $this->fail('Accessing translations relationship threw an error: ' . $e->getMessage());
        }

        $form->delete();
    }

    public function testPluginDeactivationHandlesMissingLanguageAdvanced(): void
    {

        if (is_plugin_active('language-advanced')) {
            $this->markTestSkipped('language-advanced plugin is active, cannot test graceful degradation');
        }

        try {
            $provider = new FormBuilderServiceProvider($this->app);
            $provider->boot();
            $this->assertTrue(true, 'Provider boots successfully when language-advanced is not active');
        } catch (\Exception $e) {
            $this->fail('Provider threw an error when language-advanced is not active: ' . $e->getMessage());
        }
    }

    public function testFormBuilderServiceProviderHasRegisterLanguageAdvancedMethod(): void
    {
        $provider = new FormBuilderServiceProvider($this->app);

        $this->assertTrue(
            method_exists($provider, 'registerLanguageAdvanced'),
            'FormBuilderServiceProvider should have registerLanguageAdvanced method'
        );

        $reflection = new \ReflectionClass($provider);
        $method = $reflection->getMethod('registerLanguageAdvanced');
        $this->assertTrue(
            $method->isProtected(),
            'registerLanguageAdvanced method should be protected'
        );
    }

    public function testFormBuilderModelFillableColumnsMatchExpectedFields(): void
    {
        $form = new FormBuilder();

        $expectedFillable = [
            'name',
            'code',
            'content',
            'properties',
            'actions',
            'submission',
            'status',
        ];

        $this->assertEquals(
            $expectedFillable,
            $form->getFillable(),
            'FormBuilder fillable columns should match expected fields'
        );
    }

    public function testFormBuilderModelHasCorrectCasts(): void
    {
        $form = new FormBuilder();

        $casts = $form->getCasts();

        $this->assertArrayHasKey('content', $casts);
        $this->assertArrayHasKey('properties', $casts);
        $this->assertArrayHasKey('submission', $casts);
        $this->assertArrayHasKey('status', $casts);

        $this->assertEquals('array', $casts['content']);
        $this->assertEquals('array', $casts['properties']);
        $this->assertEquals('array', $casts['submission']);
    }
}
