<?php

namespace Botble\BbFormBuilder\Tests\Feature;

use Botble\ACL\Http\Middleware\CheckPermission;
use Botble\ACL\Models\User;
use Botble\Base\Enums\BaseStatusEnum;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Illuminate\Auth\Middleware\Authenticate;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class UrlTrackingFeatureTest extends TestCase
{
    use WithFaker;
    use DatabaseTransactions;

    protected FormBuilder $form;

    protected string $formHash;

    protected function setUp(): void
    {
        parent::setUp();

        $user = User::query()->firstOrCreate(
            ['username' => 'admin_test'],
            [
                'first_name' => 'System',
                'last_name' => 'Admin',
                'email' => 'admin_test@example.com',
                'password' => bcrypt('password'),
                'super_user' => 1,
                'manage_supers' => 1,
            ]
        );

        config(['auth.guards.acl' => [
            'driver' => 'session',
            'provider' => 'users',
        ]]);

        $this->withoutMiddleware([
            \Botble\ACL\Http\Middleware\Authenticate::class,
            Authenticate::class,
            CheckPermission::class,
        ]);

        $this->createTestForm();
    }

    protected function createTestForm(): void
    {
        $uniqueCode = 'url-tracking-test-' . time() . '-' . rand(1000, 9999);

        $this->form = FormBuilder::query()->create([
            'name' => 'URL Tracking Test Form',
            'code' => $uniqueCode,
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => json_encode([
                [
                    [
                        'type' => 'text',
                        'label' => 'Name',
                        'name' => 'name',
                        'required' => true,
                    ],
                    [
                        'type' => 'email',
                        'label' => 'Email',
                        'name' => 'email',
                        'required' => true,
                    ],
                ],
            ]),
            'actions' => [
                'database' => [
                    [
                        'unique_field' => '',
                    ],
                ],
            ],
        ]);

        $this->formHash = base64_encode($this->form->id . '|' . $this->form->code);
    }

    protected function adminUser(): User
    {
        return User::query()->firstOrCreate(
            ['username' => 'admin_test'],
            [
                'first_name' => 'System',
                'last_name' => 'Admin',
                'email' => 'admin_test@example.com',
                'password' => bcrypt('password'),
                'super_user' => 1,
                'manage_supers' => 1,
            ]
        );
    }

    public function testSubmissionSavesPageUrl(): void
    {
        $pageUrl = 'https://example.com/contact-page';

        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            '_fb_page_url' => $pageUrl,
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($pageUrl, $submission->page_url);
    }

    public function testSubmissionSavesReferrerUrl(): void
    {
        $referrerUrl = 'https://google.com/search?q=contact';

        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'Jane Doe',
            'email' => 'jane@example.com',
            '_fb_referrer_url' => $referrerUrl,
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($referrerUrl, $submission->referrer_url);
    }

    public function testSubmissionSavesBothUrls(): void
    {
        $pageUrl = 'https://mysite.com/landing-page';
        $referrerUrl = 'https://facebook.com/ad/12345';

        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'Test User',
            'email' => 'test@example.com',
            '_fb_page_url' => $pageUrl,
            '_fb_referrer_url' => $referrerUrl,
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($pageUrl, $submission->page_url);
        $this->assertEquals($referrerUrl, $submission->referrer_url);
    }

    public function testSubmissionWithoutUrlTracking(): void
    {
        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'No URL User',
            'email' => 'nourl@example.com',
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertNull($submission->page_url);
        $this->assertNull($submission->referrer_url);
    }

    public function testSubmissionFallsBackToRefererHeader(): void
    {
        $refererUrl = 'https://example.com/from-referer-header';

        $response = $this->withHeaders([
            'Referer' => $refererUrl,
        ])->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'Header Test',
            'email' => 'header@example.com',
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($refererUrl, $submission->page_url);
    }

    public function testHiddenFieldOverridesRefererHeader(): void
    {
        $hiddenPageUrl = 'https://example.com/from-hidden-field';
        $refererUrl = 'https://example.com/from-referer-header';

        $response = $this->withHeaders([
            'Referer' => $refererUrl,
        ])->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'Override Test',
            'email' => 'override@example.com',
            '_fb_page_url' => $hiddenPageUrl,
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($hiddenPageUrl, $submission->page_url);
    }

    public function testInvalidUrlIsNotSaved(): void
    {
        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'Invalid URL User',
            'email' => 'invalid@example.com',
            '_fb_page_url' => 'not-a-valid-url',
            '_fb_referrer_url' => 'also-not-valid',
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertNull($submission->page_url);
        $this->assertNull($submission->referrer_url);
    }

    public function testUrlFieldsNotIncludedInSubmissionContent(): void
    {
        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'Content Test',
            'email' => 'content@example.com',
            '_fb_page_url' => 'https://example.com/page',
            '_fb_referrer_url' => 'https://example.com/referrer',
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertArrayNotHasKey('_fb_page_url', $submission->content);
        $this->assertArrayNotHasKey('_fb_referrer_url', $submission->content);
        $this->assertArrayHasKey('name', $submission->content);
        $this->assertArrayHasKey('email', $submission->content);
    }

    public function testApiSubmitSavesUrlTracking(): void
    {
        $pageUrl = 'https://external-site.com/embedded-form';
        $referrerUrl = 'https://search-engine.com/results';

        $response = $this->post(route('api.bb-form-builder.form.submit', $this->formHash), [
            'name' => 'API User',
            'email' => 'api@example.com',
            '_fb_page_url' => $pageUrl,
            '_fb_referrer_url' => $referrerUrl,
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($pageUrl, $submission->page_url);
        $this->assertEquals($referrerUrl, $submission->referrer_url);
    }

    public function testSubmissionViewDisplaysUrls(): void
    {
        $this->actingAs($this->adminUser(), 'acl');

        $submission = FormBuilderSubmission::query()->create([
            'form_id' => $this->form->id,
            'content' => ['name' => 'View Test', 'email' => 'view@test.com'],
            'ip_address' => '127.0.0.1',
            'page_url' => 'https://example.com/test-page',
            'referrer_url' => 'https://google.com/search',
        ]);

        $response = $this->get(route('bb-form-builder.submissions.show', $submission->id));

        $response->assertStatus(200);
        $response->assertSee('https://example.com/test-page');
        $response->assertSee('https://google.com/search');
    }

    public function testUrlWithQueryParametersIsSaved(): void
    {
        $pageUrl = 'https://example.com/page?utm_source=google&utm_medium=cpc&utm_campaign=summer';

        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'UTM Test',
            'email' => 'utm@example.com',
            '_fb_page_url' => $pageUrl,
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($pageUrl, $submission->page_url);
        $this->assertStringContainsString('utm_source=google', $submission->page_url);
    }

    public function testUrlWithFragmentIsSaved(): void
    {
        $pageUrl = 'https://example.com/page#contact-form';

        $response = $this->post(route('public.bb-form-builder.submit', $this->formHash), [
            'name' => 'Fragment Test',
            'email' => 'fragment@example.com',
            '_fb_page_url' => $pageUrl,
        ]);

        $response->assertStatus(200);

        $submission = FormBuilderSubmission::query()
            ->where('form_id', $this->form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals($pageUrl, $submission->page_url);
    }

    public function testFormBuilderSubmissionModelHasUrlFields(): void
    {
        $submission = new FormBuilderSubmission();

        $this->assertContains('page_url', $submission->getFillable());
        $this->assertContains('referrer_url', $submission->getFillable());
    }
}
