<?php

namespace Botble\BbFormBuilder\Tests\Feature;

use Botble\ACL\Http\Middleware\CheckPermission;
use Botble\ACL\Models\User;
use Botble\Base\Enums\BaseStatusEnum;
use Botble\BbFormBuilder\Models\FormBuilder;
use Illuminate\Auth\Middleware\Authenticate;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class FormTitleDescriptionFeatureTest extends TestCase
{
    use WithFaker;
    use DatabaseTransactions;

    protected function setUp(): void
    {
        parent::setUp();

        config(['auth.guards.acl' => [
            'driver' => 'session',
            'provider' => 'users',
        ]]);

        $this->withoutMiddleware([
            \Botble\ACL\Http\Middleware\Authenticate::class,
            Authenticate::class,
            CheckPermission::class,
        ]);
    }

    protected function adminUser(): User
    {
        return User::query()->firstOrCreate(
            ['username' => 'admin_test'],
            [
                'first_name' => 'System',
                'last_name' => 'Admin',
                'email' => 'admin_test@example.com',
                'password' => bcrypt('password'),
                'super_user' => 1,
                'manage_supers' => 1,
            ]
        );
    }

    public function testFormCanStoreAndRetrieveTitleAndDescription(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Test Form',
            'code' => 'test-form-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'text', 'label' => 'Name', 'name' => 'name'],
                ],
            ],
            'properties' => [
                'form_title' => 'Get In Touch',
                'form_description' => 'Fill out the form below and we will get back to you within 24 hours.',
                'layout' => 'horizontal',
            ],
        ]);

        $this->assertEquals('Get In Touch', $form->properties['form_title']);
        $this->assertEquals('Fill out the form below and we will get back to you within 24 hours.', $form->properties['form_description']);
    }

    public function testFormCanUpdateTitleAndDescription(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Test Form',
            'code' => 'test-form-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_title' => 'Original Title',
                'form_description' => 'Original description.',
            ],
        ]);

        $form->update([
            'properties' => [
                'form_title' => 'Updated Title',
                'form_description' => 'Updated description text.',
            ],
        ]);

        $form->refresh();

        $this->assertEquals('Updated Title', $form->properties['form_title']);
        $this->assertEquals('Updated description text.', $form->properties['form_description']);
    }

    public function testFormCanHaveOnlyTitle(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Title Only Form',
            'code' => 'title-only-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_title' => 'Contact Us',
            ],
        ]);

        $this->assertEquals('Contact Us', $form->properties['form_title']);
        $this->assertArrayNotHasKey('form_description', $form->properties);
    }

    public function testFormCanHaveOnlyDescription(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Description Only Form',
            'code' => 'desc-only-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_description' => 'Please fill out this form.',
            ],
        ]);

        $this->assertEquals('Please fill out this form.', $form->properties['form_description']);
        $this->assertArrayNotHasKey('form_title', $form->properties);
    }

    public function testFormCanHaveNeitherTitleNorDescription(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'No Title Form',
            'code' => 'no-title-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'layout' => 'horizontal',
            ],
        ]);

        $this->assertArrayNotHasKey('form_title', $form->properties);
        $this->assertArrayNotHasKey('form_description', $form->properties);
    }

    public function testFormTitleAndDescriptionAreRenderedInPublicView(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Rendered Form',
            'code' => 'rendered-form-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'text', 'label' => 'Name', 'name' => 'name'],
                ],
            ],
            'properties' => [
                'form_title' => 'Subscribe to Newsletter',
                'form_description' => 'Join our community and stay updated.',
            ],
        ]);

        $hash = base64_encode($form->id . '|' . $form->code);

        $response = $this->get(route('public.bb-form-builder.embed', $hash));

        $response->assertStatus(200);
        $response->assertSee('Subscribe to Newsletter');
        $response->assertSee('Join our community and stay updated.');
        $response->assertSee('fb-form-header');
        $response->assertSee('fb-form-title');
        $response->assertSee('fb-form-description');
    }

    public function testFormWithoutTitleDescriptionDoesNotRenderHeader(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'No Header Form',
            'code' => 'no-header-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'text', 'label' => 'Email', 'name' => 'email'],
                ],
            ],
            'properties' => [
                'layout' => 'horizontal',
            ],
        ]);

        $hash = base64_encode($form->id . '|' . $form->code);

        $response = $this->get(route('public.bb-form-builder.embed', $hash));

        $response->assertStatus(200);
        $response->assertDontSee('fb-form-header');
        $response->assertDontSee('fb-form-title');
        $response->assertDontSee('fb-form-description');
    }

    public function testFormWithOnlyTitleRendersHeaderWithTitle(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Title Only Rendered',
            'code' => 'title-only-render-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'email', 'label' => 'Email', 'name' => 'email'],
                ],
            ],
            'properties' => [
                'form_title' => 'Quick Contact',
            ],
        ]);

        $hash = base64_encode($form->id . '|' . $form->code);

        $response = $this->get(route('public.bb-form-builder.embed', $hash));

        $response->assertStatus(200);
        $response->assertSee('Quick Contact');
        $response->assertSee('fb-form-header');
        $response->assertSee('fb-form-title');
        $response->assertDontSee('fb-form-description');
    }

    public function testFormWithOnlyDescriptionRendersHeaderWithDescription(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Description Only Rendered',
            'code' => 'desc-only-render-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'email', 'label' => 'Email', 'name' => 'email'],
                ],
            ],
            'properties' => [
                'form_description' => 'We respect your privacy.',
            ],
        ]);

        $hash = base64_encode($form->id . '|' . $form->code);

        $response = $this->get(route('public.bb-form-builder.embed', $hash));

        $response->assertStatus(200);
        $response->assertSee('We respect your privacy.');
        $response->assertSee('fb-form-header');
        $response->assertDontSee('fb-form-title');
        $response->assertSee('fb-form-description');
    }

    public function testGetTranslatedPropertiesReturnsFormTitleAndDescription(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Translated Form',
            'code' => 'translated-form-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_title' => 'Contact Form',
                'form_description' => 'Please fill out this form.',
                'layout' => 'horizontal',
            ],
        ]);

        $translatedProperties = $form->getTranslatedProperties();

        $this->assertEquals('Contact Form', $translatedProperties['form_title']);
        $this->assertEquals('Please fill out this form.', $translatedProperties['form_description']);
        $this->assertEquals('horizontal', $translatedProperties['layout']);
    }

    public function testFormTitleAndDescriptionWithSpecialCharacters(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Special Chars Form',
            'code' => 'special-chars-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_title' => 'Contact Us & Get Help!',
                'form_description' => 'We\'re here to help. Questions? Ask us anything!',
            ],
        ]);

        $this->assertEquals('Contact Us & Get Help!', $form->properties['form_title']);
        $this->assertEquals('We\'re here to help. Questions? Ask us anything!', $form->properties['form_description']);
    }

    public function testFormTitleAndDescriptionInApiResponse(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'API Test Form',
            'code' => 'api-test-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'text', 'label' => 'Name', 'name' => 'name'],
                ],
            ],
            'properties' => [
                'form_title' => 'API Form Title',
                'form_description' => 'API Form Description',
            ],
        ]);

        $response = $this->get(route('api.bb-form-builder.form', $form->code));

        $response->assertStatus(200);
        $response->assertJson([
            'error' => false,
        ]);

        $responseData = $response->json('data');
        $this->assertStringContainsString('API Form Title', $responseData['html']);
        $this->assertStringContainsString('API Form Description', $responseData['html']);
        $this->assertStringContainsString('fb-form-title', $responseData['html']);
        $this->assertStringContainsString('fb-form-description', $responseData['html']);
    }

    public function testFormTitleAndDescriptionWithEmptyValues(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Empty Values Form',
            'code' => 'empty-values-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_title' => '',
                'form_description' => '',
                'layout' => 'horizontal',
            ],
        ]);

        $this->assertEquals('', $form->properties['form_title']);
        $this->assertEquals('', $form->properties['form_description']);
    }

    public function testEmptyTitleAndDescriptionDoNotRenderHeader(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Empty Header Form',
            'code' => 'empty-header-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'text', 'label' => 'Name', 'name' => 'name'],
                ],
            ],
            'properties' => [
                'form_title' => '',
                'form_description' => '',
            ],
        ]);

        $hash = base64_encode($form->id . '|' . $form->code);

        $response = $this->get(route('public.bb-form-builder.embed', $hash));

        $response->assertStatus(200);
        $response->assertDontSee('fb-form-header');
    }

    public function testFormTitleAndDescriptionWithHtmlEntitiesAreEscaped(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'HTML Entities Form',
            'code' => 'html-entities-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [
                [
                    ['type' => 'text', 'label' => 'Name', 'name' => 'name'],
                ],
            ],
            'properties' => [
                'form_title' => 'Contact <b>Us</b>',
                'form_description' => 'Fill out <script>alert("xss")</script> form',
            ],
        ]);

        $hash = base64_encode($form->id . '|' . $form->code);

        $response = $this->get(route('public.bb-form-builder.embed', $hash));

        $response->assertStatus(200);

        // HTML should be escaped to prevent XSS - verify escaped entities are present
        $response->assertSee('&lt;script&gt;', false);
        $response->assertSee('&lt;b&gt;', false);

        // The actual unescaped script tag should NOT be present
        $content = $response->getContent();
        $this->assertStringNotContainsString('<script>alert("xss")</script>', $content);
    }

    public function testFormTitleAndDescriptionWithLongText(): void
    {
        $longTitle = str_repeat('A', 255);
        $longDescription = str_repeat('B', 1000);

        $form = FormBuilder::query()->create([
            'name' => 'Long Text Form',
            'code' => 'long-text-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_title' => $longTitle,
                'form_description' => $longDescription,
            ],
        ]);

        $this->assertEquals($longTitle, $form->properties['form_title']);
        $this->assertEquals($longDescription, $form->properties['form_description']);
    }

    public function testFormTitleAndDescriptionWithUnicodeCharacters(): void
    {
        $form = FormBuilder::query()->create([
            'name' => 'Unicode Form',
            'code' => 'unicode-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => [],
            'properties' => [
                'form_title' => 'Liên hệ với chúng tôi',
                'form_description' => 'Điền vào biểu mẫu dưới đây. 日本語テスト',
            ],
        ]);

        $this->assertEquals('Liên hệ với chúng tôi', $form->properties['form_title']);
        $this->assertEquals('Điền vào biểu mẫu dưới đây. 日本語テスト', $form->properties['form_description']);
    }
}
