<?php

namespace Botble\BbFormBuilder\Tests\Feature;

use Botble\ACL\Http\Middleware\CheckPermission;
use Botble\ACL\Models\User;
use Botble\Base\Enums\BaseStatusEnum;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Illuminate\Auth\Middleware\Authenticate;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class FormBuilderFeatureTest extends TestCase
{
    use WithFaker;
    use DatabaseTransactions;

    protected function setUp(): void
    {
        parent::setUp();

        // Create an admin user and authenticate
        $user = User::query()->firstOrCreate(
            ['username' => 'admin_test'],
            [
                'first_name' => 'System',
                'last_name' => 'Admin',
                'email' => 'admin_test@example.com',
                'password' => bcrypt('password'),
                'super_user' => 1,
                'manage_supers' => 1,
            ]
        );

        // Configure ACL guard
        config(['auth.guards.acl' => [
            'driver' => 'session',
            'provider' => 'users',
        ]]);
        $this->withoutMiddleware([
            \Botble\ACL\Http\Middleware\Authenticate::class,
            Authenticate::class,
            CheckPermission::class,
        ]);
    }

    protected function adminUser()
    {
        return User::query()->firstOrCreate(
            ['username' => 'admin_test'],
            [
                'first_name' => 'System',
                'last_name' => 'Admin',
                'email' => 'admin_test@example.com',
                'password' => bcrypt('password'),
                'super_user' => 1,
                'manage_supers' => 1,
            ]
        );
    }

    public function testCanListForms()
    {
        $this->actingAs($this->adminUser(), 'acl');
        $response = $this->get(route('bb-form-builder.forms.index'));

        $response->assertStatus(200);
        $response->assertViewIs('core/table::table');
    }

    public function testCanCreateForm()
    {
        $this->actingAs($this->adminUser(), 'acl');
        $data = [
            'name' => $this->faker->name,
            'code' => $this->faker->slug,
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => json_encode([
                [
                    [
                        'type' => 'text',
                        'label' => 'Name',
                        'name' => 'name',
                    ],
                ],
            ]),
            'properties' => [
                'layout' => 'vertical',
            ],
        ];

        $response = $this->post(route('bb-form-builder.forms.store'), $data);

        $response->assertStatus(302);
        $response->assertRedirect();

        $this->assertDatabaseHas('fb_forms', [
            'name' => $data['name'],
            'code' => $data['code'],
            'status' => $data['status'],
        ]);
    }

    public function testCanUpdateForm()
    {
        $this->actingAs($this->adminUser(), 'acl');
        $uniqueCode = 'old-code-' . time();
        $form = FormBuilder::query()->create([
            'name' => 'Old Name',
            'code' => $uniqueCode,
            'status' => BaseStatusEnum::PUBLISHED,
        ]);

        $newData = [
            'name' => 'New Name',
            'code' => 'new-code-' . time(),
            'status' => BaseStatusEnum::DRAFT,
            'content' => json_encode([]),
        ];

        $response = $this->post(route('bb-form-builder.forms.update', $form), $newData);

        $response->assertStatus(302);
        $response->assertRedirect();

        $this->assertDatabaseHas('fb_forms', [
            'id' => $form->id,
            'name' => 'New Name',
            'status' => BaseStatusEnum::DRAFT,
        ]);
    }

    public function testCanDeleteForm()
    {
        $this->actingAs($this->adminUser(), 'acl');
        $form = FormBuilder::query()->create([
            'name' => 'To Delete',
            'code' => 'to-delete-' . time(),
            'status' => BaseStatusEnum::PUBLISHED,
        ]);

        $response = $this->delete(route('bb-form-builder.forms.destroy', $form->id));

        $response->assertStatus(302);

        $this->assertDatabaseMissing('fb_forms', [
            'id' => $form->id,
        ]);
    }

    public function testCanSubmitForm()
    {
        // Create a form with database action
        $uniqueCode = 'contact-form-' . time() . '-' . rand(1000, 9999);
        $form = FormBuilder::query()->create([
            'name' => 'Contact Form',
            'code' => $uniqueCode,
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => json_encode([
                [
                    [
                        'type' => 'text',
                        'label' => 'Name',
                        'name' => 'name',
                        'required' => true,
                    ],
                    [
                        'type' => 'email',
                        'label' => 'Email',
                        'name' => 'email',
                        'required' => true,
                    ],
                ],
            ]),
            'actions' => [
                'database' => [
                    [
                        'unique_field' => '',
                    ],
                ],
            ],
        ]);

        // Generate hash
        $hash = base64_encode($form->id . '|' . $form->code);

        // Submit data
        $submissionData = [
            'name' => 'John Doe',
            'email' => 'john@example.com',
        ];

        $response = $this->post(route('public.bb-form-builder.submit', $hash), $submissionData);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);

        // Verify submission was created
        $submission = FormBuilderSubmission::query()->where('form_id', $form->id)
            ->orderBy('id', 'desc')
            ->first();

        $this->assertNotNull($submission);
        $this->assertEquals('John Doe', $submission->content['name']);
        $this->assertEquals('john@example.com', $submission->content['email']);
    }

    public function testCanGetFormByCode()
    {
        $uniqueCode = 'api-form-' . time() . '-' . rand(1000, 9999);
        $form = FormBuilder::query()->create([
            'name' => 'API Form',
            'code' => $uniqueCode,
            'status' => BaseStatusEnum::PUBLISHED,
            'content' => json_encode([]),
        ]);

        $response = $this->get(route('api.bb-form-builder.form', $form->code));

        $response->assertStatus(200);
        $response->assertJson([
            'error' => false,
            'data' => [
                'form_hash' => base64_encode($form->id . '|' . $form->code),
            ],
        ]);
    }
}
