<?php

namespace Botble\BbFormBuilder\Tables;

use Botble\Base\Facades\BaseHelper;
use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\Actions\DeleteAction;
use Botble\Table\Actions\ViewAction;
use Botble\Table\BulkActions\DeleteBulkAction;
use Botble\Table\Columns\Column;
use Botble\Table\Columns\CreatedAtColumn;
use Botble\Table\Columns\FormattedColumn;
use Botble\Table\Columns\IdColumn;
use Botble\Table\HeaderActions\HeaderAction;
use Illuminate\Database\Eloquent\Builder;

class SubmissionTable extends TableAbstract
{
    public function setup(): void
    {
        $this
            ->model(FormBuilderSubmission::class)
            ->setOption('id', 'fb-form-submission-table')
            ->addHeaderAction(
                HeaderAction::make('export-form-submissions')
                    ->label(trans('plugins/bb-form-builder::form.export.heading'))
                    ->icon('ti ti-download')
                    ->url(route('tools.data-synchronize.export.form-submissions.index'))
                    ->permission('bb-form-builder.submissions.export')
            )
            ->addActions([
                ViewAction::make()
                    ->route('bb-form-builder.submissions.show')
                    ->iconOnly(),
                DeleteAction::make()->route('bb-form-builder.submissions.destroy'),
            ])
            ->addColumns([
                IdColumn::make(),
                FormattedColumn::make('form_name')
                    ->label(trans('plugins/bb-form-builder::form.submissions.form'))
                    ->searchable(false)
                    ->orderable(false)
                    ->getValueUsing(fn (FormattedColumn $column) => $column->getItem()->form?->name ?? '-'),
                FormattedColumn::make('identifier')
                    ->label(trans('plugins/bb-form-builder::form.submissions.identifier'))
                    ->searchable(false)
                    ->orderable(false)
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        if (! empty($item->unique_identifier)) {
                            return BaseHelper::clean($item->unique_identifier);
                        }

                        $content = $item->content;

                        if (is_array($content) && ! empty($content)) {
                            $firstValue = reset($content);

                            if (is_array($firstValue)) {
                                $firstValue = implode(', ', $firstValue);
                            }

                            return BaseHelper::clean($firstValue);
                        }

                        return '-';
                    }),
                Column::make('ip_address')
                    ->label(trans('plugins/bb-form-builder::form.submissions.ip_address')),
                CreatedAtColumn::make()
                    ->label(trans('plugins/bb-form-builder::form.submissions.submitted_at')),
            ])
            ->addBulkAction(
                DeleteBulkAction::make()->permission('bb-form-builder.submissions.destroy'),
            )
            ->queryUsing(function (Builder $query): void {
                $query->with('form')
                    ->select([
                        'id',
                        'form_id',
                        'content',
                        'unique_identifier',
                        'ip_address',
                        'created_at',
                    ]);
            });
    }
}
