<?php

namespace Botble\BbFormBuilder\Tables;

use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\Actions\DeleteAction;
use Botble\Table\Actions\ViewAction;
use Botble\Table\BulkActions\DeleteBulkAction;
use Botble\Table\BulkChanges\CreatedAtBulkChange;
use Botble\Table\Columns\Column;
use Botble\Table\Columns\CreatedAtColumn;
use Botble\Table\Columns\IdColumn;
use Illuminate\Database\Eloquent\Builder;

class FormSubmissionTable extends TableAbstract
{
    protected ?FormBuilder $form = null;

    public function forForm(FormBuilder $form): self
    {
        $this->form = $form;

        return $this;
    }

    public function setup(): void
    {
        $this
            ->model(FormBuilderSubmission::class)
            ->setOption('id', 'fb-form-submission-table')
            ->addActions([
                ViewAction::make()
                    ->route('bb-form-builder.submissions.show')
                    ->iconOnly(),
                DeleteAction::make()->route('bb-form-builder.submissions.destroy'),
            ])
            ->addColumns([
                IdColumn::make(),
                Column::make('ip_address')
                    ->label(trans('plugins/bb-form-builder::form.submissions.ip_address')),
                Column::make('user_agent')
                    ->label(trans('plugins/bb-form-builder::form.submissions.user_agent'))
                    ->width(200)
                    ->getValueUsing(function (Column $column) {
                        $userAgent = $column->getItem()->user_agent;
                        if (strlen($userAgent) > 50) {
                            return '<span title="' . e($userAgent) . '">' . e(substr($userAgent, 0, 50)) . '...</span>';
                        }

                        return e($userAgent);
                    }),
                CreatedAtColumn::make()
                    ->label(trans('plugins/bb-form-builder::form.submissions.submitted_at')),
            ])
            ->addBulkAction(
                DeleteBulkAction::make()->permission('bb-form-builder.submissions.destroy'),
            )
            ->addBulkChanges([
                CreatedAtBulkChange::make(),
            ]);
    }

    public function query(): Builder
    {
        $query = $this->getModel()
            ->query()
            ->select([
                'id',
                'form_id',
                'ip_address',
                'user_agent',
                'created_at',
            ]);

        if ($this->form) {
            $query->where('form_id', $this->form->id);
        }

        return $this->applyScopes($query);
    }
}
