<?php

namespace Botble\BbFormBuilder\Tables;

use Botble\Base\Enums\BaseStatusEnum;
use Botble\Base\Facades\Html;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\Actions\DeleteAction;
use Botble\Table\Actions\EditAction;
use Botble\Table\Actions\ViewAction;
use Botble\Table\BulkActions\DeleteBulkAction;
use Botble\Table\BulkChanges\StatusBulkChange;
use Botble\Table\Columns\CreatedAtColumn;
use Botble\Table\Columns\FormattedColumn;
use Botble\Table\Columns\IdColumn;
use Botble\Table\Columns\NameColumn;
use Botble\Table\Columns\StatusColumn;
use Botble\Table\HeaderActions\CreateHeaderAction;
use Botble\Table\HeaderActions\HeaderAction;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Validation\Rule;

class FormBuilderTable extends TableAbstract
{
    public function setup(): void
    {
        $this
            ->model(FormBuilder::class)
            ->setOption('id', 'fb-form-table')
            ->setView('plugins/bb-form-builder::forms.list')
            ->addHeaderActions([
                CreateHeaderAction::make()
                    ->route('bb-form-builder.forms.create')
                    ->permission('bb-form-builder.forms.create'),
                HeaderAction::make('import-form')
                    ->label(trans('plugins/bb-form-builder::form.export_import.import_button'))
                    ->icon('ti ti-upload')
                    ->url('#')
                    ->permission('bb-form-builder.forms.create')
                    ->attributes(['class' => 'btn-import-form']),
            ])
            ->addActions([
                ViewAction::make()->route('bb-form-builder.forms.show'),
                EditAction::make()->route('bb-form-builder.forms.edit'),
                DeleteAction::make()->route('bb-form-builder.forms.destroy'),
            ])
            ->addColumns([
                IdColumn::make(),
                NameColumn::make()
                    ->route('bb-form-builder.forms.edit'),
                FormattedColumn::make('code')
                    ->label(trans('plugins/bb-form-builder::form.forms.short_code'))
                    ->alignStart()
                    ->getValueUsing(function (FormattedColumn $column) {
                        $value = $column->getItem()->code;

                        if (! function_exists('shortcode')) {
                            return $value;
                        }

                        return shortcode()->generateShortcode('bb-form-builder', ['code' => $value]);
                    })
                    ->renderUsing(fn (FormattedColumn $column) => Html::tag('code', $column->getValue()))
                    ->copyable()
                    ->copyableState(function (FormattedColumn $column) {
                        $value = $column->getItem()->code;

                        if (! function_exists('shortcode')) {
                            return $value;
                        }

                        return shortcode()->generateShortcode('bb-form-builder', ['code' => $value]);
                    }),
                FormattedColumn::make('submissions_count')
                    ->label(trans('plugins/bb-form-builder::form.forms.submissions_count'))
                    ->orderable(false)
                    ->searchable(false)
                    ->getValueUsing(function (FormattedColumn $column) {
                        return number_format($column->getItem()->submissions_count);
                    })
                    ->withEmptyState(),
                StatusColumn::make(),
                CreatedAtColumn::make(),
            ])
            ->addBulkAction(
                DeleteBulkAction::make()->permission('bb-form-builder.forms.destroy'),
            )
            ->addBulkChange(
                StatusBulkChange::make()
                    ->choices(BaseStatusEnum::labels())
                    ->validate(['required', Rule::in(BaseStatusEnum::values())]),
            )
            ->queryUsing(function (Builder $query): void {
                $query
                    ->select([
                        'id',
                        'name',
                        'code',
                        'status',
                        'created_at',
                    ])
                    ->withCount('submissions');
            });
    }
}
