<?php

namespace Botble\BbFormBuilder\Providers;

use Botble\Base\Enums\BaseStatusEnum;
use Botble\Base\Forms\FieldOptions\SelectFieldOption;
use Botble\Base\Forms\Fields\SelectField;
use Botble\Base\Supports\ServiceProvider;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\Shortcode\Facades\Shortcode;
use Botble\Shortcode\Forms\ShortcodeForm;
use Illuminate\Support\Arr;

class HookServiceProvider extends ServiceProvider
{
    public function boot(): void
    {
        $this->registerShortcode();
    }

    protected function registerShortcode(): void
    {
        Shortcode::register(
            'bb-form-builder',
            trans('plugins/bb-form-builder::form.shortcode.title'),
            trans('plugins/bb-form-builder::form.shortcode.description'),
            [$this, 'renderForm']
        );

        Shortcode::ignoreCaches(['bb-form-builder']);

        Shortcode::setAdminConfig('bb-form-builder', function (array $attributes) {
            $forms = FormBuilder::query()
                ->where('status', BaseStatusEnum::PUBLISHED)
                ->pluck('name', 'code')
                ->all();

            $styles = [
                '' => trans('plugins/bb-form-builder::form.shortcode.use_form_default'),
                'default' => trans('plugins/bb-form-builder::form.styles.default'),
                'modern' => trans('plugins/bb-form-builder::form.styles.modern'),
                'minimal' => trans('plugins/bb-form-builder::form.styles.minimal'),
                'bold' => trans('plugins/bb-form-builder::form.styles.bold'),
                'glass' => trans('plugins/bb-form-builder::form.styles.glass'),
            ];

            return ShortcodeForm::createFromArray($attributes)
                ->add(
                    'code',
                    SelectField::class,
                    SelectFieldOption::make()
                        ->label(trans('plugins/bb-form-builder::form.shortcode.form'))
                        ->choices($forms)
                        ->selected(Arr::get($attributes, 'code'))
                )
                ->add(
                    'style',
                    SelectField::class,
                    SelectFieldOption::make()
                        ->label(trans('plugins/bb-form-builder::form.shortcode.style'))
                        ->choices($styles)
                        ->selected(Arr::get($attributes, 'style'))
                );
        });
    }

    public function renderForm($shortcode): string
    {
        $code = $shortcode->code;

        if (empty($code)) {
            return '';
        }

        $form = FormBuilder::query()
            ->where('code', $code)
            ->where('status', BaseStatusEnum::PUBLISHED)
            ->first();

        if (! $form) {
            return '';
        }

        $styleOverride = $shortcode->style ?: null;
        $disableAutoTrigger = filter_var($shortcode->disable_auto_trigger ?? false, FILTER_VALIDATE_BOOLEAN);

        return view('plugins/bb-form-builder::forms.render', compact('form', 'styleOverride', 'disableAutoTrigger'))->render();
    }
}
