<?php

namespace Botble\BbFormBuilder\Models;

use Botble\Base\Casts\SafeContent;
use Botble\Base\Enums\BaseStatusEnum;
use Botble\Base\Models\BaseModel;
use Botble\Language\Facades\Language;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class FormBuilder extends BaseModel
{
    protected $table = 'fb_forms';

    protected $fillable = [
        'name',
        'code',
        'content',
        'properties',
        'actions',
        'submission',
        'status',
    ];

    protected $casts = [
        'name' => SafeContent::class,
        'status' => BaseStatusEnum::class,
        'content' => 'array',
        'properties' => 'array',
        'actions' => 'array',
        'submission' => 'array',
    ];

    protected static function booted(): void
    {
        static::saving(function (FormBuilder $form): void {
            if (empty($form->code)) {
                $form->code = Str::slug($form->name) . '-' . Str::random(6);
            }

            // Merge new properties with existing ones to preserve step_titles and other nested data
            if ($form->exists && $form->isDirty('properties')) {
                $originalProperties = $form->getRawOriginal('properties');
                $originalProperties = is_string($originalProperties) ? json_decode($originalProperties, true) : $originalProperties;
                $originalProperties = is_array($originalProperties) ? $originalProperties : [];

                $newProperties = $form->properties ?? [];
                $newProperties = is_array($newProperties) ? $newProperties : [];

                // Deep merge properties
                $form->properties = array_replace_recursive($originalProperties, $newProperties);
            }
        });
    }

    public function submissions(): HasMany
    {
        return $this->hasMany(FormBuilderSubmission::class, 'form_id');
    }

    protected function hash(): Attribute
    {
        return Attribute::get(fn () => base64_encode($this->id . '|' . $this->code));
    }

    protected function isActive(): Attribute
    {
        return Attribute::get(fn () => $this->status == BaseStatusEnum::PUBLISHED);
    }

    protected function hasCaptcha(): Attribute
    {
        return Attribute::get(fn () => $this->properties['has_captcha'] ?? false);
    }

    protected function hasMathCaptcha(): Attribute
    {
        return Attribute::get(fn () => $this->properties['has_math_captcha'] ?? false);
    }
    public function getOriginalContent(): array
    {
        $content = $this->getRawOriginal('content');

        if (is_string($content)) {
            $content = json_decode($content, true);
        }

        return is_array($content) ? $content : [];
    }

    public function getFieldsForStep(string $step): array
    {
        $content = $this->getOriginalContent();

        return $content[$step] ?? [];
    }

    public function getAllFields(): array
    {
        $fields = [];
        $content = $this->getOriginalContent();

        foreach ($content as $stepFields) {
            if (is_array($stepFields)) {
                foreach ($stepFields as $field) {
                    if (is_array($field) && isset($field['name'])) {
                        $fields[] = $field;
                    }
                }
            }
        }

        return $fields;
    }

    public function getFieldsLabel(): array
    {
        $labels = [];

        foreach ($this->getAllFields() as $field) {
            if (is_array($field) && isset($field['name'])) {
                $labels[$field['name']] = $field['label'] ?? $field['name'];
            }
        }

        return $labels;
    }

    public function getTranslatedContent(): array
    {
        $originalContent = $this->getOriginalContent();

        $translation = $this->getActiveTranslation();

        if (! $translation || ! $translation->content) {
            return $originalContent;
        }

        $translatedContent = is_string($translation->content)
            ? json_decode($translation->content, true)
            : $translation->content;

        if (! is_array($translatedContent)) {
            return $originalContent;
        }

        return $this->mergeTranslatedContent($originalContent, $translatedContent);
    }

    protected function mergeTranslatedContent(array $original, array $translated): array
    {
        $result = $original;

        foreach ($result as $step => $stepFields) {
            if (! is_array($stepFields)) {
                continue;
            }

            foreach ($stepFields as $index => $field) {
                if (! is_array($field) || ! isset($field['name'])) {
                    continue;
                }

                $fieldName = $field['name'];
                $translatedField = $this->findTranslatedField($translated, $step, $fieldName);

                if (! $translatedField) {
                    continue;
                }

                if (isset($translatedField['label'])) {
                    $result[$step][$index]['label'] = $translatedField['label'];
                }

                if (isset($translatedField['placeholder'])) {
                    $result[$step][$index]['placeholder'] = $translatedField['placeholder'];
                }

                if (isset($translatedField['options']) && isset($field['options'])) {
                    $result[$step][$index]['options'] = $this->mergeTranslatedOptions(
                        $field['options'],
                        $translatedField['options']
                    );
                }
            }
        }

        return $result;
    }

    protected function findTranslatedField(array $translated, string $step, string $fieldName): ?array
    {
        if (isset($translated[$step]) && is_array($translated[$step])) {
            foreach ($translated[$step] as $field) {
                if (is_array($field) && ($field['name'] ?? '') === $fieldName) {
                    return $field;
                }
            }
        }

        foreach ($translated as $stepFields) {
            if (! is_array($stepFields)) {
                continue;
            }

            foreach ($stepFields as $field) {
                if (is_array($field) && ($field['name'] ?? '') === $fieldName) {
                    return $field;
                }
            }
        }

        return null;
    }

    protected function mergeTranslatedOptions(array $originalOptions, array $translatedOptions): array
    {
        $translatedMap = [];
        foreach ($translatedOptions as $opt) {
            if (isset($opt['value'])) {
                $translatedMap[$opt['value']] = $opt['label'] ?? $opt['value'];
            }
        }

        $result = $originalOptions;
        foreach ($result as $index => $opt) {
            if (isset($opt['value']) && isset($translatedMap[$opt['value']])) {
                $result[$index]['label'] = $translatedMap[$opt['value']];
            }
        }

        return $result;
    }

    public function getTranslatedProperties(): array
    {
        $translation = $this->getActiveTranslation();

        // Get original properties, ensuring it's an array
        $originalProps = $this->getRawOriginal('properties');
        $originalProps = is_string($originalProps) ? json_decode($originalProps, true) : $originalProps;
        $originalProps = is_array($originalProps) ? $originalProps : [];

        if ($translation && $translation->properties) {
            $props = is_string($translation->properties)
                ? json_decode($translation->properties, true)
                : $translation->properties;

            if (is_array($props)) {
                return array_merge($originalProps, $props);
            }
        }

        return $originalProps;
    }

    public function getTranslatedSubmission(): array
    {
        $translation = $this->getActiveTranslation();

        // Get original submission, ensuring it's an array
        $originalSub = $this->getRawOriginal('submission');
        $originalSub = is_string($originalSub) ? json_decode($originalSub, true) : $originalSub;
        $originalSub = is_array($originalSub) ? $originalSub : [];

        if ($translation && $translation->submission) {
            $sub = is_string($translation->submission)
                ? json_decode($translation->submission, true)
                : $translation->submission;

            if (is_array($sub)) {
                return array_merge($originalSub, $sub);
            }
        }

        return $originalSub;
    }

    protected function getActiveTranslation()
    {
        if (! is_plugin_active('language') || ! is_plugin_active('language-advanced')) {
            return null;
        }

        $locale = is_in_admin()
            ? (Language::getCurrentAdminLocaleCode() ?: Language::getRefLang())
            : Language::getCurrentLocaleCode();

        if (! $locale) {
            return null;
        }

        $defaultLocale = Language::getDefaultLocaleCode();
        if ($locale === $defaultLocale) {
            return null;
        }

        if ($this->relationLoaded('translations')) {
            return $this->translations->firstWhere('lang_code', $locale);
        }

        return $this->translations()->where('lang_code', $locale)->first();
    }
}
