<?php

namespace Botble\BbFormBuilder\Http\Requests;

use Botble\Base\Enums\BaseStatusEnum;
use Botble\Support\Http\Requests\Request;
use Illuminate\Validation\Rule;

class FormBuilderRequest extends Request
{
    protected function prepareForValidation(): void
    {
        $content = $this->input('content');
        if (is_string($content)) {
            $this->merge(['content' => json_decode($content, true) ?: []]);
        }

        $actions = $this->input('actions');
        if (is_string($actions)) {
            $this->merge(['actions' => json_decode($actions, true) ?: []]);
        }
    }

    public function rules(): array
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'code' => [
                'nullable',
                'string',
                'max:120',
                Rule::unique('fb_forms', 'code')->ignore($this->route('form')),
            ],
            'content' => ['nullable', 'array'],
            'properties' => ['nullable', 'array'],
            'properties.form_title' => ['nullable', 'string', 'max:255'],
            'properties.form_description' => ['nullable', 'string', 'max:2000'],
            'properties.has_captcha' => ['nullable'],
            'properties.has_math_captcha' => ['nullable'],
            'properties.layout' => ['nullable', 'string', Rule::in(['horizontal', 'vertical'])],
            'properties.form_size' => ['nullable', 'string', Rule::in(['large', 'medium', 'small', 'full', 'xxl', 'xl', 'lg', 'md', 'sm'])],
            'properties.form_alignment' => ['nullable', 'string', Rule::in(['start', 'center', 'end'])],
            'properties.form_style' => ['nullable', 'string', Rule::in(['default', 'modern', 'minimal', 'bold', 'glass'])],
            'properties.form_margin' => ['nullable', 'string', 'max:100'],
            'properties.form_padding' => ['nullable', 'string', 'max:100'],
            'properties.allow_embed' => ['nullable'],
            'properties.step_titles' => ['nullable', 'array'],
            'properties.step_titles.*' => ['nullable', 'string', 'max:255'],
            'properties.step_conditions' => ['nullable', 'array'],
            'properties.step_conditions.*' => ['nullable', 'array'],
            'properties.step_conditions.*.field' => ['nullable', 'string', 'max:255'],
            'properties.step_conditions.*.operator' => ['nullable', 'string', Rule::in(['checked', 'not_checked', 'equals', 'not_equals'])],
            'properties.step_conditions.*.value' => ['nullable', 'string', 'max:255'],
            'properties.display_mode' => ['nullable', 'string', Rule::in(['inline', 'popup'])],
            'properties.popup_trigger' => ['nullable', 'string', Rule::in(['button', 'time_delay', 'scroll', 'exit_intent'])],
            'properties.popup_button_text' => ['nullable', 'string', 'max:255'],
            'properties.popup_delay' => ['nullable', 'integer', 'min:1', 'max:300'],
            'properties.popup_scroll_percentage' => ['nullable', 'integer', 'min:1', 'max:100'],
            'properties.popup_title' => ['nullable', 'string', 'max:255'],
            'properties.popup_max_width' => ['nullable', 'string', Rule::in(['sm', 'md', 'lg', 'xl'])],
            'properties.popup_close_on_overlay' => ['nullable'],
            'properties.popup_show_close_button' => ['nullable'],
            'actions' => ['nullable', 'array'],
            'submission' => ['nullable', 'array'],
            'submission.on_success' => ['nullable', 'array'],
            'submission.on_success.action' => ['nullable', 'string', Rule::in(['show_message', 'redirect_to'])],
            'submission.on_success.content' => ['nullable', 'string', 'max:2000'],
            'submission.on_failure' => ['nullable', 'array'],
            'submission.on_failure.action' => ['nullable', 'string', Rule::in(['show_message', 'redirect_to'])],
            'submission.on_failure.content' => ['nullable', 'string', 'max:2000'],
            'status' => ['required', Rule::in(BaseStatusEnum::values())],
        ];
    }

    public function attributes(): array
    {
        return [
            'name' => trans('plugins/bb-form-builder::form.forms.name'),
            'code' => trans('plugins/bb-form-builder::form.forms.code'),
            'status' => trans('plugins/bb-form-builder::form.forms.status'),
        ];
    }
}
