<?php

namespace Botble\BbFormBuilder\Http\Controllers;

use Botble\Base\Enums\BaseStatusEnum;
use Botble\Base\Supports\Breadcrumb;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Illuminate\Support\Facades\DB;

class ReportsController extends BaseFormBuilderController
{
    protected function breadcrumb(): Breadcrumb
    {
        return parent::breadcrumb()
            ->add(trans('plugins/bb-form-builder::form.menu.title'))
            ->add(trans('plugins/bb-form-builder::form.menu.reports'), route('bb-form-builder.reports.index'));
    }

    public function index()
    {
        $this->pageTitle(trans('plugins/bb-form-builder::form.reports.title'));

        $now = now();
        $startOfWeek = $now->copy()->startOfWeek();
        $endOfWeek = $now->copy()->endOfWeek();
        $startOfMonth = $now->copy()->startOfMonth();
        $endOfMonth = $now->copy()->endOfMonth();

        $forms = FormBuilder::query()
            ->select(['id', 'name', 'status', 'created_at'])
            ->withCount('submissions')
            ->latest()
            ->get();

        $overallSubmissions = FormBuilderSubmission::query()
            ->selectRaw(
                <<<'SQL'
COUNT(*) as total,
SUM(CASE WHEN DATE(created_at) = ? THEN 1 ELSE 0 END) as today,
SUM(CASE WHEN created_at BETWEEN ? AND ? THEN 1 ELSE 0 END) as this_week,
SUM(CASE WHEN created_at BETWEEN ? AND ? THEN 1 ELSE 0 END) as this_month
SQL,
                [
                    $now->toDateString(),
                    $startOfWeek,
                    $endOfWeek,
                    $startOfMonth,
                    $endOfMonth,
                ]
            )
            ->first();

        $totalForms = $forms->count();
        $totalSubmissions = (int) ($overallSubmissions?->total ?? 0);
        $activeForms = $forms->filter(
            fn (FormBuilder $form) => ($form->status?->getValue()) === BaseStatusEnum::PUBLISHED
        )->count();
        $todaySubmissions = (int) ($overallSubmissions?->today ?? 0);
        $weekSubmissions = (int) ($overallSubmissions?->this_week ?? 0);
        $monthSubmissions = (int) ($overallSubmissions?->this_month ?? 0);

        $dailySubmissions = FormBuilderSubmission::query()
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
            ->where('created_at', '>=', $now->copy()->subDays(29)->startOfDay())
            ->groupBy('date')
            ->orderBy('date')
            ->pluck('count', 'date')
            ->toArray();

        $chartDates = collect(range(0, 29))
            ->map(fn (int $index) => $now->copy()->subDays(29 - $index)->toDateString());

        $chartData = $chartDates
            ->mapWithKeys(fn (string $date) => [$date => (int) ($dailySubmissions[$date] ?? 0)])
            ->toArray();

        $topForms = $forms
            ->sortByDesc('submissions_count')
            ->take(5);

        $recentSubmissions = FormBuilderSubmission::query()
            ->with(['form:id,name,status'])
            ->latest()
            ->limit(10)
            ->get();

        $formSubmissionAggregates = FormBuilderSubmission::query()
            ->selectRaw(
                <<<'SQL'
form_id,
COUNT(*) as total,
SUM(CASE WHEN DATE(created_at) = ? THEN 1 ELSE 0 END) as today,
SUM(CASE WHEN created_at BETWEEN ? AND ? THEN 1 ELSE 0 END) as week,
SUM(CASE WHEN created_at BETWEEN ? AND ? THEN 1 ELSE 0 END) as month
SQL,
                [
                    $now->toDateString(),
                    $startOfWeek,
                    $endOfWeek,
                    $startOfMonth,
                    $endOfMonth,
                ]
            )
            ->whereIn('form_id', $forms->pluck('id'))
            ->groupBy('form_id')
            ->get()
            ->keyBy('form_id');

        $formStats = $forms->map(function (FormBuilder $form) use ($formSubmissionAggregates) {
            $aggregates = $formSubmissionAggregates->get($form->id);

            return [
                'id' => $form->id,
                'name' => $form->name,
                'status' => $form->status,
                'total' => (int) ($aggregates?->total ?? 0),
                'today' => (int) ($aggregates?->today ?? 0),
                'week' => (int) ($aggregates?->week ?? 0),
                'month' => (int) ($aggregates?->month ?? 0),
            ];
        });

        return view('plugins/bb-form-builder::reports.index', compact(
            'totalForms',
            'totalSubmissions',
            'activeForms',
            'todaySubmissions',
            'weekSubmissions',
            'monthSubmissions',
            'chartData',
            'topForms',
            'recentSubmissions',
            'formStats'
        ));
    }
}
