<?php

namespace Botble\BbFormBuilder\Http\Controllers;

use Botble\Base\Enums\BaseStatusEnum;
use Botble\Base\Http\Controllers\BaseController;
use Botble\BbFormBuilder\Classes\SubmissionHandler;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\Language\Facades\Language;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PublicController extends BaseController
{
    public function embed(string $hash)
    {
        $form = $this->getFormByHash($hash);

        abort_unless($form, 404);

        abort_unless($form->properties['allow_embed'] ?? true, 403, trans('plugins/bb-form-builder::form.messages.embed_disabled'));

        return view('plugins/bb-form-builder::forms.embed', compact('form'));
    }

    public function embedByCode(string $code)
    {
        $query = FormBuilder::query()
            ->where('code', $code)
            ->where('status', BaseStatusEnum::PUBLISHED);

        $this->applyTranslationEagerLoading($query);

        $form = $query->first();

        abort_unless($form, 404);

        abort_unless($form->properties['allow_embed'] ?? true, 403, trans('plugins/bb-form-builder::form.messages.embed_disabled'));

        return view('plugins/bb-form-builder::forms.embed', compact('form'));
    }

    public function submit(string $hash, Request $request, SubmissionHandler $submissionHandler)
    {
        $form = $this->getFormByHash($hash);

        if (! $form) {
            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('plugins/bb-form-builder::form.messages.form_not_found'));
        }

        if (! $form->is_active) {
            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('plugins/bb-form-builder::form.messages.form_inactive'));
        }

        return $submissionHandler->submit($request, $form);
    }

    public function apiSubmit(string $hash, Request $request, SubmissionHandler $submissionHandler): JsonResponse
    {
        $form = $this->getFormByHash($hash);

        if (! $form) {
            return $this->corsResponse([
                'error' => true,
                'message' => trans('plugins/bb-form-builder::form.messages.form_not_found'),
            ], 404);
        }

        if (! ($form->properties['allow_embed'] ?? true)) {
            return $this->corsResponse([
                'error' => true,
                'message' => trans('plugins/bb-form-builder::form.messages.embed_disabled'),
            ], 403);
        }

        if (! $form->is_active) {
            return $this->corsResponse([
                'error' => true,
                'message' => trans('plugins/bb-form-builder::form.messages.form_inactive'),
            ], 400);
        }

        $response = $submissionHandler->submit($request, $form);

        if ($response instanceof JsonResponse) {
            $data = $response->getData(true);
            $status = $response->getStatusCode();

            return $this->corsResponse($data, $status);
        }

        return $this->corsResponse([
            'error' => false,
            'message' => trans('plugins/bb-form-builder::form.messages.submission_success'),
        ]);
    }

    protected function corsResponse(array $data, int $status = 200): JsonResponse
    {
        return response()->json($data, $status)->withHeaders([
            'Access-Control-Allow-Origin' => '*',
            'Access-Control-Allow-Methods' => 'POST, OPTIONS',
            'Access-Control-Allow-Headers' => 'Content-Type, X-Requested-With',
        ]);
    }

    public function getFormByCode(string $code): JsonResponse
    {
        $query = FormBuilder::query()
            ->where('code', $code)
            ->where('status', BaseStatusEnum::PUBLISHED);

        $this->applyTranslationEagerLoading($query);

        $form = $query->first();

        if (! $form) {
            return $this->corsResponse([
                'error' => true,
                'message' => trans('plugins/bb-form-builder::form.messages.form_not_found'),
            ], 404);
        }

        if (! ($form->properties['allow_embed'] ?? true)) {
            return $this->corsResponse([
                'error' => true,
                'message' => trans('plugins/bb-form-builder::form.messages.embed_disabled'),
            ], 403);
        }

        $html = view('plugins/bb-form-builder::forms.render', [
            'form' => $form,
            'isEmbed' => true,
        ])->render();

        return $this->corsResponse([
            'error' => false,
            'data' => [
                'html' => $html,
                'form_hash' => $form->hash,
                'submit_url' => route('api.bb-form-builder.form.submit', $form->hash),
                'has_phone_field' => $this->formHasPhoneField($form),
                'has_multiple_steps' => is_array($form->content) && count($form->content) > 1 && ! isset($form->content[0]),
            ],
        ]);
    }

    public function getFormAssets(string $code): JsonResponse
    {
        $form = FormBuilder::query()
            ->where('code', $code)
            ->where('status', BaseStatusEnum::PUBLISHED)
            ->first();

        if (! $form) {
            return $this->corsResponse([
                'error' => true,
                'message' => trans('plugins/bb-form-builder::form.messages.form_not_found'),
            ], 404);
        }

        $hasPhoneField = $this->formHasPhoneField($form);
        $hasMultipleSteps = is_array($form->content) && count($form->content) > 1 && ! isset($form->content[0]);

        $css = [
            asset('vendor/core/plugins/bb-form-builder/css/form-public.css'),
        ];

        $js = [
            asset('vendor/core/plugins/bb-form-builder/js/form-public.js'),
        ];

        if ($hasPhoneField) {
            $css[] = asset('vendor/core/core/base/libraries/intl-tel-input/css/intlTelInput.min.css');
            $js[] = asset('vendor/core/core/base/libraries/intl-tel-input/js/intlTelInput.min.js');
        }

        if ($hasMultipleSteps) {
            $css[] = asset('vendor/core/plugins/bb-form-builder/css/smart_wizard.min.css');
            $js[] = asset('vendor/core/plugins/bb-form-builder/js/jquery.smartWizard.min.js');
        }

        return $this->corsResponse([
            'error' => false,
            'data' => [
                'css' => $css,
                'js' => $js,
            ],
        ]);
    }

    protected function formHasPhoneField(FormBuilder $form): bool
    {
        foreach ($form->getAllFields() as $field) {
            if (($field['type'] ?? '') === 'phone') {
                return true;
            }
        }

        return false;
    }

    protected function getFormByHash(string $hash): ?FormBuilder
    {
        $decoded = base64_decode($hash);

        if (! $decoded || ! str_contains($decoded, '|')) {
            return null;
        }

        [$id, $code] = explode('|', $decoded);

        $query = FormBuilder::query()
            ->where('id', $id)
            ->where('code', $code);

        $this->applyTranslationEagerLoading($query);

        return $query->first();
    }

    protected function applyTranslationEagerLoading(Builder $query): void
    {
        if (! is_plugin_active('language-advanced') || ! is_plugin_active('language')) {
            return;
        }

        $currentLocale = Language::getCurrentLocale();
        $defaultLocale = Language::getDefaultLocale();

        if ($currentLocale === $defaultLocale) {
            return;
        }

        $query->with(['translations' => function ($q) use ($currentLocale): void {
            $q->where('lang_code', $currentLocale);
        }]);
    }
}
