<?php

namespace Botble\BbFormBuilder\Http\Controllers;

use Botble\Base\Facades\Assets;
use Botble\Base\Http\Actions\DeleteResourceAction;
use Botble\Base\Supports\Breadcrumb;
use Botble\BbFormBuilder\Forms\FormBuilderForm;
use Botble\BbFormBuilder\Http\Requests\FormBuilderRequest;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Tables\FormBuilderTable;
use Botble\BbFormBuilder\Tables\FormSubmissionTable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\StreamedResponse;

class FormBuilderController extends BaseFormBuilderController
{
    protected function breadcrumb(): Breadcrumb
    {
        return parent::breadcrumb()
            ->add(trans('plugins/bb-form-builder::form.menu.title'))
            ->add(trans('plugins/bb-form-builder::form.forms.title'), route('bb-form-builder.forms.index'));
    }

    public function index(FormBuilderTable $table)
    {
        $this->pageTitle(trans('plugins/bb-form-builder::form.forms.title'));

        Assets::addScriptsDirectly('vendor/core/plugins/bb-form-builder/js/import-form.js')
            ->addScripts(['blockui']);

        return $table->renderTable();
    }

    public function show(FormBuilder $form, FormSubmissionTable $table)
    {
        $table->forForm($form);

        if (request()->ajax() || request()->wantsJson()) {
            return $table->renderTable();
        }

        $this->pageTitle(trans('plugins/bb-form-builder::form.forms.view', ['name' => $form->name]));

        $form->loadCount('submissions');

        $stats = $this->getFormStatistics($form);

        return view('plugins/bb-form-builder::forms.show', compact('form', 'stats', 'table'));
    }

    protected function getFormStatistics(FormBuilder $form): array
    {
        $totalSubmissions = $form->submissions()->count();

        $todaySubmissions = $form->submissions()
            ->whereDate('created_at', today())
            ->count();

        $weekSubmissions = $form->submissions()
            ->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])
            ->count();

        $monthSubmissions = $form->submissions()
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();

        $dailySubmissions = $form->submissions()
            ->select(DB::raw('DATE(created_at) as date'), DB::raw('COUNT(*) as count'))
            ->where('created_at', '>=', now()->subDays(30))
            ->groupBy('date')
            ->orderBy('date')
            ->pluck('count', 'date')
            ->toArray();

        $chartData = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i)->format('Y-m-d');
            $chartData[$date] = $dailySubmissions[$date] ?? 0;
        }

        return [
            'total' => $totalSubmissions,
            'today' => $todaySubmissions,
            'week' => $weekSubmissions,
            'month' => $monthSubmissions,
            'chart_data' => $chartData,
        ];
    }

    public function create(): string
    {
        $this->pageTitle(trans('plugins/bb-form-builder::form.forms.create'));

        return FormBuilderForm::create()->renderForm();
    }

    public function store(FormBuilderRequest $request)
    {
        $form = FormBuilderForm::create();
        $form->setRequest($request)->saveOnlyValidatedData();

        $activeTab = $request->input('active_tab');

        return $this
            ->httpResponse()
            ->setNextUrl(route('bb-form-builder.forms.edit', $form->getModel()) . ($activeTab ? $activeTab : ''))
            ->withCreatedSuccessMessage();
    }

    public function edit(FormBuilder $form): string
    {
        $this->pageTitle(trans('core/base::forms.edit_item', ['name' => $form->name]));

        return FormBuilderForm::createFromModel($form)->renderForm();
    }

    public function update(FormBuilder $form, FormBuilderRequest $request)
    {
        FormBuilderForm::createFromModel($form)->setRequest($request)->saveOnlyValidatedData();

        $activeTab = $request->input('active_tab');

        return $this
            ->httpResponse()
            ->setPreviousUrl(route('bb-form-builder.forms.index'))
            ->setNextUrl(route('bb-form-builder.forms.edit', $form->id) . ($activeTab ? $activeTab : ''))
            ->withUpdatedSuccessMessage();
    }

    public function destroy(FormBuilder $form)
    {
        return DeleteResourceAction::make($form);
    }

    public function saveTranslation(FormBuilder $form, Request $request)
    {
        $request->validate([
            'ref_lang' => ['required', 'string', 'max:20'],
            'name' => ['nullable', 'string', 'max:250'],
            'field_labels' => ['nullable', 'array'],
            'field_labels.*' => ['nullable', 'string', 'max:250'],
            'field_placeholders' => ['nullable', 'array'],
            'field_placeholders.*' => ['nullable', 'string', 'max:500'],
            'field_options' => ['nullable', 'array'],
            'properties' => ['nullable', 'array'],
            'submission' => ['nullable', 'array'],
        ]);

        $refLang = $request->input('ref_lang');

        if (! is_plugin_active('language-advanced')) {
            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('plugins/bb-form-builder::form.translation.invalid_language'));
        }

        try {
            $originalContent = $form->content;
            if (is_string($originalContent)) {
                $originalContent = json_decode($originalContent, true) ?? [];
            }

            $translatedContent = $this->buildTranslatedContent(
                $originalContent ?? [],
                $request->input('field_labels', []),
                $request->input('field_placeholders', []),
                $request->input('field_options', [])
            );

            $translatedProperties = array_filter($request->input('properties', []), fn ($v) => $v !== '' && $v !== null);

            $submissionInput = $request->input('submission', []);
            $translatedSubmission = [];

            $originalSubmission = $form->submission;
            if (is_string($originalSubmission)) {
                $originalSubmission = json_decode($originalSubmission, true) ?? [];
            }

            if (! empty($submissionInput['on_success']['content'])) {
                $translatedSubmission['on_success'] = [
                    'action' => $originalSubmission['on_success']['action'] ?? 'show_message',
                    'content' => $submissionInput['on_success']['content'],
                ];
            }

            if (! empty($submissionInput['on_failure']['content'])) {
                $translatedSubmission['on_failure'] = [
                    'action' => $originalSubmission['on_failure']['action'] ?? 'show_message',
                    'content' => $submissionInput['on_failure']['content'],
                ];
            }

            DB::table('fb_forms_translations')->updateOrInsert(
                ['lang_code' => $refLang, 'fb_forms_id' => $form->id],
                [
                    'name' => $request->input('name'),
                    'content' => ! empty($translatedContent) ? json_encode($translatedContent) : null,
                    'properties' => ! empty($translatedProperties) ? json_encode($translatedProperties) : null,
                    'submission' => ! empty($translatedSubmission) ? json_encode($translatedSubmission) : null,
                ]
            );

            return $this
                ->httpResponse()
                ->setNextUrl(route('bb-form-builder.forms.edit', $form->id) . '?ref_lang=' . $refLang)
                ->withUpdatedSuccessMessage();
        } catch (\Exception $e) {
            report($e);

            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('core/base::notices.error'));
        }
    }

    protected function buildTranslatedContent(array $originalContent, array $labels, array $placeholders, array $options): array
    {
        $translated = [];

        foreach ($originalContent as $step => $stepFields) {
            if (! is_array($stepFields)) {
                continue;
            }

            $translated[$step] = [];

            foreach ($stepFields as $field) {
                if (! is_array($field) || ! isset($field['name'])) {
                    continue;
                }

                $fieldName = $field['name'];
                $translatedField = [];

                if (! empty($labels[$fieldName])) {
                    $translatedField['name'] = $fieldName;
                    $translatedField['label'] = $labels[$fieldName];
                }

                if (! empty($placeholders[$fieldName])) {
                    $translatedField['name'] = $fieldName;
                    $translatedField['placeholder'] = $placeholders[$fieldName];
                }

                if (! empty($options[$fieldName]) && ! empty($field['options'])) {
                    $translatedOptions = [];
                    foreach ($field['options'] as $opt) {
                        $optValue = $opt['value'] ?? '';
                        $translatedOptions[] = [
                            'value' => $optValue,
                            'label' => $options[$fieldName][$optValue] ?? $opt['label'] ?? $optValue,
                        ];
                    }
                    $translatedField['name'] = $fieldName;
                    $translatedField['options'] = $translatedOptions;
                }

                if (! empty($translatedField)) {
                    $translated[$step][] = $translatedField;
                }
            }
        }

        return array_filter($translated, fn ($step) => ! empty($step));
    }

    public function export(FormBuilder $form): StreamedResponse
    {
        $exportData = [
            'version' => '1.0',
            'exported_at' => now()->toIso8601String(),
            'exported_from' => url('/'),
            'form' => [
                'name' => $form->name,
                'code' => $form->code,
                'content' => $form->content,
                'properties' => $form->properties,
                'actions' => $form->actions,
                'submission' => $form->submission,
                'status' => $form->status->getValue(),
            ],
        ];

        $filename = Str::slug($form->name) . '-' . now()->format('Y-m-d') . '.json';

        return response()->streamDownload(function () use ($exportData): void {
            echo json_encode($exportData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }, $filename, [
            'Content-Type' => 'application/json',
        ]);
    }

    public function import(Request $request)
    {
        $data = (array) $request->input('json_data', []);

        if (empty($data)) {
            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('plugins/bb-form-builder::form.export_import.invalid_file'));
        }

        try {
            if (! isset($data['form']) || ! is_array($data['form'])) {
                return $this
                    ->httpResponse()
                    ->setError()
                    ->setMessage(trans('plugins/bb-form-builder::form.export_import.invalid_format'));
            }

            $formData = $data['form'];

            if (empty($formData['name']) || empty($formData['content'])) {
                return $this
                    ->httpResponse()
                    ->setError()
                    ->setMessage(trans('plugins/bb-form-builder::form.export_import.invalid_format'));
            }

            $existingCode = FormBuilder::query()->where('code', $formData['code'] ?? '')->exists();
            $newCode = $existingCode ? Str::slug($formData['name']) . '-' . Str::random(6) : ($formData['code'] ?? null);

            $form = FormBuilder::query()->create([
                'name' => $formData['name'] . ' (Imported)',
                'code' => $newCode,
                'content' => $formData['content'],
                'properties' => $formData['properties'] ?? [],
                'actions' => $formData['actions'] ?? [],
                'submission' => $formData['submission'] ?? [],
                'status' => $formData['status'] ?? 'published',
            ]);

            return $this
                ->httpResponse()
                ->setNextUrl(route('bb-form-builder.forms.edit', $form))
                ->setMessage(trans('plugins/bb-form-builder::form.export_import.import_success'));
        } catch (\Exception $e) {
            report($e);

            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('plugins/bb-form-builder::form.export_import.import_error'));
        }
    }
}
