<?php

namespace Botble\BbFormBuilder\Forms;

use Botble\Base\Enums\BaseStatusEnum;
use Botble\Base\Forms\FieldOptions\NameFieldOption;
use Botble\Base\Forms\FieldOptions\OnOffFieldOption;
use Botble\Base\Forms\FieldOptions\StatusFieldOption;
use Botble\Base\Forms\FieldOptions\TextFieldOption;
use Botble\Base\Forms\Fields\OnOffCheckboxField;
use Botble\Base\Forms\Fields\SelectField;
use Botble\Base\Forms\Fields\TextField;
use Botble\Base\Forms\FormAbstract;
use Botble\BbFormBuilder\Http\Requests\FormBuilderRequest;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\Captcha\Facades\Captcha;
use Botble\Language\Facades\Language;

class FormBuilderForm extends FormAbstract
{
    public function setup(): void
    {
        $this
            ->model(FormBuilder::class)
            ->setValidatorClass(FormBuilderRequest::class)
            ->setFormOption('class', 'bb-form-builder-form')
            ->contentOnly()
            ->add(
                'name',
                TextField::class,
                NameFieldOption::make()
                    ->required()
                    ->toArray()
            )
            ->add(
                'code',
                TextField::class,
                TextFieldOption::make()
                    ->label(trans('plugins/bb-form-builder::form.forms.code'))
                    ->helperText(trans('plugins/bb-form-builder::form.forms.code_help'))
                    ->toArray()
            )
            ->add(
                'status',
                SelectField::class,
                StatusFieldOption::make()
                    ->choices(BaseStatusEnum::labels())
                    ->toArray()
            )
            ->setBreakFieldPoint('status');

        if (is_plugin_active('captcha')) {
            $this->add(
                'properties[has_captcha]',
                OnOffCheckboxField::class,
                OnOffFieldOption::make()
                    ->label(trans('plugins/bb-form-builder::form.builder.has_captcha'))
                    ->value($this->getModel()->properties['has_captcha'] ?? false)
                    ->toArray()
            );

            if (Captcha::mathCaptchaEnabled()) {
                $this->add(
                    'properties[has_math_captcha]',
                    OnOffCheckboxField::class,
                    OnOffFieldOption::make()
                        ->label(trans('plugins/bb-form-builder::form.builder.has_math_captcha'))
                        ->value($this->getModel()->properties['has_math_captcha'] ?? false)
                        ->toArray()
                );
            }
        }
    }

    public function renderForm(array $options = [], bool $showStart = true, bool $showFields = true, bool $showEnd = true): string
    {
        $model = $this->getModel();

        if ($this->isTranslationMode()) {
            return $this->renderTranslationForm();
        }

        $actions = config('plugins.bb-form-builder.form-builder.actions', []);

        return view('plugins/bb-form-builder::forms.create-edit', [
            'form' => $this,
            'model' => $model,
            'actions' => $actions,
        ])->render();
    }

    protected function isTranslationMode(): bool
    {
        if (! is_plugin_active('language-advanced') || ! is_plugin_active('language')) {
            return false;
        }

        $model = $this->getModel();

        if (! $model || ! $model->getKey()) {
            return false;
        }

        $refLang = Language::getRefLang();

        if (! $refLang) {
            return false;
        }

        return $refLang !== Language::getDefaultLocaleCode();
    }

    protected function renderTranslationForm(): string
    {
        $model = $this->getModel();
        $refLang = Language::getRefLang();

        $translation = $model->translations()
            ->where('lang_code', $refLang)
            ->first();

        $languages = Language::getActiveLanguage(['lang_code', 'lang_flag', 'lang_name', 'lang_locale']);
        $currentLanguage = $languages->firstWhere('lang_code', $refLang)
            ?? $languages->firstWhere('lang_locale', $refLang);

        abort_unless($currentLanguage, 404, trans('plugins/bb-form-builder::form.translation.invalid_language'));

        return view('plugins/bb-form-builder::forms.translation', [
            'form' => $this,
            'model' => $model,
            'translation' => $translation,
            'refLang' => $refLang,
            'languages' => $languages,
            'currentLanguage' => $currentLanguage,
        ])->render();
    }
}
