<?php

namespace Botble\BbFormBuilder\Exporters;

use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Botble\DataSynchronize\Exporter\ExportColumn;
use Botble\DataSynchronize\Exporter\ExportCounter;
use Botble\DataSynchronize\Exporter\Exporter;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;

class FormSubmissionExporter extends Exporter
{
    protected ?int $limit = null;

    protected ?string $formId = null;

    protected ?string $startDate = null;

    protected ?string $endDate = null;

    protected bool $contentOnly = false;

    protected ?array $resolvedContentColumns = null;

    public function getLabel(): string
    {
        return trans('plugins/bb-form-builder::form.submissions.title');
    }

    public function getHeading(): string
    {
        return trans('plugins/bb-form-builder::form.export.heading');
    }

    public function columns(): array
    {
        return [
            ...($this->contentOnly ? $this->contentColumns() : [
                ExportColumn::make('id'),
                ExportColumn::make('form_id'),
                ExportColumn::make('form_name'),
                ExportColumn::make('form_code'),
                ExportColumn::make('unique_identifier'),
                ExportColumn::make('ip_address'),
                ExportColumn::make('user_agent'),
                ExportColumn::make('created_at'),
                ExportColumn::make('content'),
            ]),
        ];
    }

    public function setLimit(?int $limit): static
    {
        $this->limit = $limit;

        return $this;
    }

    public function setFormId(?string $formId): static
    {
        $this->formId = $formId;

        return $this;
    }

    public function setDateRange(?string $startDate, ?string $endDate): static
    {
        $this->startDate = $startDate;
        $this->endDate = $endDate;

        return $this;
    }

    public function contentOnly(bool $contentOnly = true): static
    {
        $this->contentOnly = $contentOnly;

        return $this;
    }

    protected function contentColumns(): array
    {
        $keys = FormBuilderSubmission::query()
            ->when($this->formId, fn (Builder $query) => $query->where('form_id', $this->formId))
            ->when($this->startDate, fn (Builder $query) => $query->whereDate('created_at', '>=', $this->startDate))
            ->when($this->endDate, fn (Builder $query) => $query->whereDate('created_at', '<=', $this->endDate))
            ->limit(2000)
            ->pluck('content')
            ->flatMap(function ($content) {
                return array_keys((array) $content);
            })
            ->filter()
            ->unique()
            ->values();

        if ($keys->isEmpty()) {
            return [
                ExportColumn::make('content'),
            ];
        }

        return $keys->map(fn ($key) => ExportColumn::make($key))->all();
    }

    protected function applyFilters(Builder $query): void
    {
        if ($this->formId) {
            $query->where('form_id', $this->formId);
        }

        if ($this->startDate) {
            $query->whereDate('created_at', '>=', $this->startDate);
        }

        if ($this->endDate) {
            $query->whereDate('created_at', '<=', $this->endDate);
        }

        if ($this->limit) {
            $query->latest()->limit($this->limit);
        } else {
            $query->latest();
        }
    }

    public function counters(): array
    {
        $query = FormBuilderSubmission::query();

        $this->applyFilters($query);

        return [
            ExportCounter::make()
                ->label(trans('plugins/bb-form-builder::form.export.total'))
                ->value($query->count()),
        ];
    }

    public function hasDataToExport(): bool
    {
        return FormBuilderSubmission::query()->exists();
    }

    public function collection(): Collection
    {
        $query = FormBuilderSubmission::query()->with('form');

        $this->applyFilters($query);

        if ($this->contentOnly) {
            $this->resolvedContentColumns ??= $this->contentColumns();
        }

        return $query->get()->transform(function (FormBuilderSubmission $submission) {
            $form = $submission->form;

            if ($this->contentOnly) {
                $row = [];

                foreach ($this->resolvedContentColumns as $column) {
                    $key = $column->getName();
                    $value = $submission->content[$key] ?? null;
                    $row[$key] = is_array($value) ? implode(', ', $value) : $value;
                }

                return $row;
            }

            return [
                'id' => $submission->id,
                'form_id' => $submission->form_id,
                'form_name' => $form?->name,
                'form_code' => $form?->code,
                'unique_identifier' => $submission->unique_identifier,
                'ip_address' => $submission->ip_address,
                'user_agent' => $submission->user_agent,
                'created_at' => $submission->created_at?->toDateTimeString(),
                'content' => json_encode($submission->content),
            ];
        });
    }

    protected function getView(): string
    {
        return 'plugins/bb-form-builder::submissions.export';
    }
}
