<?php

namespace Botble\BbFormBuilder\Database\Seeders;

use Botble\Base\Enums\BaseStatusEnum;
use Botble\Base\Supports\BaseSeeder;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Illuminate\Support\Facades\DB;

class FormBuilderSeeder extends BaseSeeder
{
    public function run(): void
    {
        $this->truncateTables();

        $forms = [
            [
                'name' => 'Contact Form',
                'code' => 'contact-form',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'text',
                            'name' => 'name',
                            'label' => 'Name',
                            'placeholder' => 'Name',
                            'required' => true,
                            'maxlength' => 40,
                            'width' => '6',
                        ],
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => 'Email',
                            'placeholder' => 'Email',
                            'required' => true,
                            'maxlength' => 80,
                            'width' => '6',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'address',
                            'label' => 'Address',
                            'placeholder' => 'Address',
                            'required' => false,
                            'maxlength' => 500,
                            'width' => '12',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'phone',
                            'label' => 'Phone',
                            'placeholder' => 'Phone',
                            'required' => false,
                            'width' => '6',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'subject',
                            'label' => 'Subject',
                            'placeholder' => 'Subject',
                            'required' => false,
                            'maxlength' => 500,
                            'width' => '6',
                        ],
                        [
                            'type' => 'textarea',
                            'name' => 'message',
                            'label' => 'Message',
                            'placeholder' => 'Message',
                            'required' => true,
                            'rows' => 5,
                            'maxlength' => 10000,
                            'width' => '12',
                        ],
                        [
                            'type' => 'checkbox-group',
                            'name' => 'agree_terms',
                            'label' => 'Terms and Privacy Policy',
                            'hide_label' => true,
                            'required' => true,
                            'width' => '12',
                            'options' => [
                                [
                                    'value' => 'agree',
                                    'label' => 'I agree to the Terms and Privacy Policy',
                                ],
                            ],
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => false,
                    'submit_button_text' => 'Send',
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [],
                    ],
                    'email' => [
                        'email_1' => [
                            'to' => 'admin@example.com',
                            'subject' => 'New Contact Form Submission from [name]',
                            'body' => "You have received a new contact form submission.\n\nName: [name]\nEmail: [email]\nAddress: [address]\nPhone: [phone]\nSubject: [subject]\n\nMessage:\n[message]",
                        ],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Thank you for contacting us! We will get back to you within 24 hours.',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Something went wrong. Please try again later.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Customer Feedback Survey',
                'code' => 'customer-feedback',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'text',
                            'name' => 'name',
                            'label' => 'Your Name',
                            'placeholder' => 'Enter your name',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => 'Email Address',
                            'placeholder' => 'Enter your email',
                            'required' => true,
                            'width' => '6',
                        ],
                    ],
                    'step-2' => [
                        [
                            'type' => 'starRating',
                            'name' => 'overall_rating',
                            'label' => 'Overall Experience',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'radio-group',
                            'name' => 'recommend',
                            'label' => 'Would you recommend us to others?',
                            'required' => true,
                            'width' => '6',
                            'options' => [
                                ['value' => 'definitely', 'label' => 'Definitely'],
                                ['value' => 'probably', 'label' => 'Probably'],
                                ['value' => 'not_sure', 'label' => 'Not Sure'],
                                ['value' => 'probably_not', 'label' => 'Probably Not'],
                                ['value' => 'definitely_not', 'label' => 'Definitely Not'],
                            ],
                        ],
                        [
                            'type' => 'checkbox-group',
                            'name' => 'improvements',
                            'label' => 'What areas could we improve?',
                            'required' => false,
                            'width' => '12',
                            'options' => [
                                ['value' => 'customer_service', 'label' => 'Customer Service'],
                                ['value' => 'product_quality', 'label' => 'Product Quality'],
                                ['value' => 'pricing', 'label' => 'Pricing'],
                                ['value' => 'website', 'label' => 'Website Experience'],
                                ['value' => 'delivery', 'label' => 'Delivery Speed'],
                            ],
                        ],
                    ],
                    'step-3' => [
                        [
                            'type' => 'textarea',
                            'name' => 'comments',
                            'label' => 'Additional Comments',
                            'placeholder' => 'Tell us more about your experience...',
                            'required' => false,
                            'rows' => 5,
                            'width' => '12',
                        ],
                        [
                            'type' => 'checkbox-group',
                            'name' => 'agree_terms',
                            'label' => 'Terms and Privacy Policy',
                            'hide_label' => true,
                            'required' => true,
                            'width' => '12',
                            'options' => [
                                [
                                    'value' => 'agree',
                                    'label' => 'I agree to the Terms and Privacy Policy',
                                ],
                            ],
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => false,
                    'layout' => 'vertical',
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Thank you for your valuable feedback! We appreciate your time.',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'An error occurred. Please try again.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Job Application Form',
                'code' => 'job-application',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'text',
                            'name' => 'full_name',
                            'label' => 'Full Name',
                            'placeholder' => 'Enter your full name',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => 'Email Address',
                            'placeholder' => 'Enter your email',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'phone',
                            'name' => 'phone',
                            'label' => 'Phone Number',
                            'placeholder' => 'Enter your phone number',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'linkedin',
                            'label' => 'LinkedIn Profile URL',
                            'placeholder' => 'https://linkedin.com/in/your-profile',
                            'required' => false,
                            'width' => '6',
                        ],
                    ],
                    'step-2' => [
                        [
                            'type' => 'select',
                            'name' => 'position',
                            'label' => 'Position Applied For',
                            'placeholder' => 'Select a position',
                            'required' => true,
                            'width' => '6',
                            'options' => [
                                ['value' => 'developer', 'label' => 'Software Developer'],
                                ['value' => 'designer', 'label' => 'UI/UX Designer'],
                                ['value' => 'manager', 'label' => 'Project Manager'],
                                ['value' => 'marketing', 'label' => 'Marketing Specialist'],
                                ['value' => 'other', 'label' => 'Other'],
                            ],
                        ],
                        [
                            'type' => 'select',
                            'name' => 'experience',
                            'label' => 'Years of Experience',
                            'placeholder' => 'Select your experience',
                            'required' => true,
                            'width' => '6',
                            'options' => [
                                ['value' => '0-1', 'label' => '0-1 years'],
                                ['value' => '1-3', 'label' => '1-3 years'],
                                ['value' => '3-5', 'label' => '3-5 years'],
                                ['value' => '5-10', 'label' => '5-10 years'],
                                ['value' => '10+', 'label' => '10+ years'],
                            ],
                        ],
                        [
                            'type' => 'date',
                            'name' => 'start_date',
                            'label' => 'Earliest Start Date',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'salary_expectation',
                            'label' => 'Salary Expectation',
                            'placeholder' => 'Enter your expected salary',
                            'required' => false,
                            'width' => '6',
                        ],
                    ],
                    'step-3' => [
                        [
                            'type' => 'file',
                            'name' => 'resume',
                            'label' => 'Upload Resume',
                            'required' => true,
                            'accept' => '.pdf,.doc,.docx',
                            'width' => '12',
                        ],
                        [
                            'type' => 'textarea',
                            'name' => 'cover_letter',
                            'label' => 'Cover Letter',
                            'placeholder' => 'Tell us why you would be a great fit...',
                            'required' => false,
                            'rows' => 6,
                            'width' => '12',
                        ],
                        [
                            'type' => 'checkbox-group',
                            'name' => 'agree_terms',
                            'label' => 'Terms and Privacy Policy',
                            'hide_label' => true,
                            'required' => true,
                            'width' => '12',
                            'options' => [
                                [
                                    'value' => 'agree',
                                    'label' => 'I agree to the Terms and Privacy Policy',
                                ],
                            ],
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => true,
                    'layout' => 'vertical',
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [],
                    ],
                    'email' => [
                        'email_1' => [
                            'to' => 'hr@example.com',
                            'subject' => 'New Job Application: [position] - [full_name]',
                            'body' => "New job application received.\n\nName: [full_name]\nEmail: [email]\nPhone: [phone]\nLinkedIn: [linkedin]\nPosition: [position]\nExperience: [experience]\nStart Date: [start_date]\nSalary Expectation: [salary_expectation]\n\nCover Letter:\n[cover_letter]",
                        ],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Thank you for your application! Our HR team will review it and contact you soon.',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Failed to submit your application. Please try again.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Newsletter Subscription',
                'code' => 'newsletter',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => '',
                            'placeholder' => 'Enter your email address...',
                            'required' => true,
                            'width' => '12',
                        ],
                        [
                            'type' => 'checkbox-group',
                            'name' => 'agree_terms',
                            'label' => 'Terms and Privacy Policy',
                            'hide_label' => true,
                            'required' => true,
                            'width' => '12',
                            'options' => [
                                [
                                    'value' => 'agree',
                                    'label' => 'I agree to the Terms and Privacy Policy',
                                ],
                            ],
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => false,
                    'submit_button_text' => 'Subscribe',
                    'layout' => 'inline',
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [
                            'unique_field' => 'email',
                        ],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Thank you for subscribing! You will receive our latest updates.',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Subscription failed. Please try again.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Event Registration',
                'code' => 'event-registration',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'text',
                            'name' => 'first_name',
                            'label' => 'First Name',
                            'placeholder' => 'Enter your first name',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'last_name',
                            'label' => 'Last Name',
                            'placeholder' => 'Enter your last name',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => 'Email Address',
                            'placeholder' => 'Enter your email',
                            'required' => true,
                            'width' => '6',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'company',
                            'label' => 'Company/Organization',
                            'placeholder' => 'Enter your company name',
                            'required' => false,
                            'width' => '6',
                        ],
                    ],
                    'step-2' => [
                        [
                            'type' => 'select',
                            'name' => 'ticket_type',
                            'label' => 'Ticket Type',
                            'placeholder' => 'Select ticket type',
                            'required' => true,
                            'width' => '6',
                            'options' => [
                                ['value' => 'standard', 'label' => 'Standard ($49)'],
                                ['value' => 'vip', 'label' => 'VIP ($99)'],
                                ['value' => 'group', 'label' => 'Group (5+ people, $39 each)'],
                            ],
                        ],
                        [
                            'type' => 'number',
                            'name' => 'attendees',
                            'label' => 'Number of Attendees',
                            'placeholder' => '1',
                            'required' => true,
                            'width' => '6',
                            'min' => 1,
                            'max' => 10,
                        ],
                        [
                            'type' => 'radio-group',
                            'name' => 'dietary',
                            'label' => 'Dietary Requirements',
                            'required' => true,
                            'width' => '12',
                            'options' => [
                                ['value' => 'none', 'label' => 'None'],
                                ['value' => 'vegetarian', 'label' => 'Vegetarian'],
                                ['value' => 'vegan', 'label' => 'Vegan'],
                                ['value' => 'gluten_free', 'label' => 'Gluten-Free'],
                                ['value' => 'other', 'label' => 'Other (please specify in comments)'],
                            ],
                        ],
                        [
                            'type' => 'textarea',
                            'name' => 'special_requests',
                            'label' => 'Special Requests or Comments',
                            'placeholder' => 'Any special requirements...',
                            'required' => false,
                            'width' => '12',
                            'rows' => 3,
                        ],
                        [
                            'type' => 'checkbox-group',
                            'name' => 'agree_terms',
                            'label' => 'Terms and Privacy Policy',
                            'hide_label' => true,
                            'required' => true,
                            'width' => '12',
                            'options' => [
                                [
                                    'value' => 'agree',
                                    'label' => 'I agree to the Terms and Privacy Policy',
                                ],
                            ],
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => true,
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [
                            'unique_field' => 'email',
                        ],
                    ],
                    'email' => [
                        'email_1' => [
                            'to' => 'events@example.com',
                            'subject' => 'New Event Registration: [first_name] [last_name]',
                            'body' => "New event registration received.\n\nName: [first_name] [last_name]\nEmail: [email]\nCompany: [company]\nTicket Type: [ticket_type]\nAttendees: [attendees]\nDietary: [dietary]\n\nSpecial Requests:\n[special_requests]",
                        ],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Registration successful! Check your email for confirmation details.',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Registration failed. Please try again or contact us.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Contact Us Popup (Button Click)',
                'code' => 'popup-contact-button',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'text',
                            'name' => 'name',
                            'label' => 'Your Name',
                            'placeholder' => 'Enter your name',
                            'required' => true,
                            'width' => '12',
                        ],
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => 'Email Address',
                            'placeholder' => 'Enter your email',
                            'required' => true,
                            'width' => '12',
                        ],
                        [
                            'type' => 'textarea',
                            'name' => 'message',
                            'label' => 'Message',
                            'placeholder' => 'How can we help you?',
                            'required' => true,
                            'rows' => 4,
                            'width' => '12',
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => false,
                    'submit_button_text' => 'Send Message',
                    'display_mode' => 'popup',
                    'popup_trigger' => 'button',
                    'popup_button_text' => 'Contact Us',
                    'popup_title' => 'Get in Touch',
                    'popup_max_width' => 'sm',
                    'popup_close_on_overlay' => true,
                    'popup_show_close_button' => true,
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Thank you! We will get back to you soon.',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Something went wrong. Please try again.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Newsletter Popup (Time Delay)',
                'code' => 'popup-newsletter-delay',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'html',
                            'name' => 'intro_text',
                            'label' => '',
                            'html_content' => '<p style="text-align: center; margin-bottom: 1rem;">Subscribe to our newsletter and get <strong>10% off</strong> your first order!</p>',
                            'width' => '12',
                        ],
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => '',
                            'placeholder' => 'Enter your email address',
                            'required' => true,
                            'width' => '12',
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => false,
                    'submit_button_text' => 'Subscribe Now',
                    'display_mode' => 'popup',
                    'popup_trigger' => 'time_delay',
                    'popup_delay' => 5,
                    'popup_title' => 'Stay Updated!',
                    'popup_max_width' => 'sm',
                    'popup_close_on_overlay' => true,
                    'popup_show_close_button' => true,
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [
                            'unique_field' => 'email',
                        ],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Welcome aboard! Check your inbox for your discount code.',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Subscription failed. Please try again.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Feedback Popup (Scroll)',
                'code' => 'popup-feedback-scroll',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'html',
                            'name' => 'intro_text',
                            'label' => '',
                            'html_content' => '<p style="text-align: center;">We noticed you\'ve been exploring our site. We\'d love to hear your thoughts!</p>',
                            'width' => '12',
                        ],
                        [
                            'type' => 'starRating',
                            'name' => 'rating',
                            'label' => 'How would you rate your experience?',
                            'required' => true,
                            'width' => '12',
                        ],
                        [
                            'type' => 'textarea',
                            'name' => 'feedback',
                            'label' => 'Any suggestions?',
                            'placeholder' => 'Tell us what you think...',
                            'required' => false,
                            'rows' => 3,
                            'width' => '12',
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => false,
                    'submit_button_text' => 'Submit Feedback',
                    'display_mode' => 'popup',
                    'popup_trigger' => 'scroll',
                    'popup_scroll_percentage' => 50,
                    'popup_title' => 'Quick Feedback',
                    'popup_max_width' => 'sm',
                    'popup_close_on_overlay' => true,
                    'popup_show_close_button' => true,
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Thank you for your feedback!',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Something went wrong. Please try again.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
            [
                'name' => 'Exit Intent Popup',
                'code' => 'popup-exit-intent',
                'content' => [
                    'step-1' => [
                        [
                            'type' => 'html',
                            'name' => 'intro_text',
                            'label' => '',
                            'html_content' => '<div style="text-align: center;"><h4 style="margin-bottom: 0.5rem;">Wait! Before you go...</h4><p>Get our exclusive guide delivered straight to your inbox!</p></div>',
                            'width' => '12',
                        ],
                        [
                            'type' => 'text',
                            'name' => 'name',
                            'label' => 'Your Name',
                            'placeholder' => 'Enter your name',
                            'required' => true,
                            'width' => '12',
                        ],
                        [
                            'type' => 'email',
                            'name' => 'email',
                            'label' => 'Email Address',
                            'placeholder' => 'Enter your email',
                            'required' => true,
                            'width' => '12',
                        ],
                    ],
                ],
                'properties' => [
                    'has_captcha' => false,
                    'submit_button_text' => 'Get Free Guide',
                    'display_mode' => 'popup',
                    'popup_trigger' => 'exit_intent',
                    'popup_title' => '',
                    'popup_max_width' => 'sm',
                    'popup_close_on_overlay' => true,
                    'popup_show_close_button' => true,
                ],
                'actions' => [
                    'database' => [
                        'db_1' => [
                            'unique_field' => 'email',
                        ],
                    ],
                ],
                'submission' => [
                    'on_success' => [
                        'action' => 'show_message',
                        'content' => 'Check your email for the guide!',
                    ],
                    'on_failure' => [
                        'action' => 'show_message',
                        'content' => 'Something went wrong. Please try again.',
                    ],
                ],
                'status' => BaseStatusEnum::PUBLISHED,
            ],
        ];

        foreach ($forms as $form) {
            FormBuilder::query()->create($form);
        }

        $this->seedSubmissions();
    }

    protected function seedSubmissions(): void
    {
        $contactForm = FormBuilder::query()->where('code', 'contact-form')->first();
        $feedbackForm = FormBuilder::query()->where('code', 'customer-feedback')->first();
        $jobForm = FormBuilder::query()->where('code', 'job-application')->first();
        $newsletterForm = FormBuilder::query()->where('code', 'newsletter')->first();
        $eventForm = FormBuilder::query()->where('code', 'event-registration')->first();

        $userAgents = [
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Mozilla/5.0 (iPhone; CPU iPhone OS 17_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/17.0 Mobile/15E148 Safari/604.1',
            'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:121.0) Gecko/20100101 Firefox/121.0',
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/17.0 Safari/605.1.15',
        ];

        $contactSubmissions = [
            [
                'name' => 'John Smith',
                'email' => 'john.smith@example.com',
                'address' => '123 Main Street, New York, NY 10001',
                'phone' => '+1 555-123-4567',
                'subject' => 'General Inquiry',
                'message' => 'I would like to know more about your services and pricing plans. Could you please send me a detailed brochure?',
            ],
            [
                'name' => 'Sarah Johnson',
                'email' => 'sarah.j@example.com',
                'address' => '',
                'phone' => '+1 555-987-6543',
                'subject' => 'Technical Support',
                'message' => 'I am having trouble logging into my account. I have tried resetting my password but still cannot access it.',
            ],
            [
                'name' => 'Michael Brown',
                'email' => 'mbrown@example.com',
                'address' => '456 Oak Avenue, Los Angeles, CA 90001',
                'phone' => '',
                'subject' => 'Sales Inquiry',
                'message' => 'We are interested in purchasing your enterprise plan for our team of 50 employees. Please contact me to discuss pricing.',
            ],
            [
                'name' => 'Emily Davis',
                'email' => 'emily.davis@example.com',
                'address' => '789 Pine Road, Chicago, IL 60601',
                'phone' => '+1 555-456-7890',
                'subject' => 'Partnership',
                'message' => 'I am a journalist writing an article about your company. Would it be possible to arrange an interview with your CEO?',
            ],
            [
                'name' => 'David Wilson',
                'email' => 'dwilson@example.com',
                'address' => '',
                'phone' => '+1 555-321-0987',
                'subject' => 'Bug Report',
                'message' => 'The mobile app keeps crashing when I try to upload images. I am using an iPhone 15 with the latest iOS version.',
            ],
        ];

        foreach ($contactSubmissions as $index => $submission) {
            FormBuilderSubmission::query()->create([
                'form_id' => $contactForm->id,
                'content' => $submission,
                'unique_identifier' => null,
                'ip_address' => '192.168.1.' . ($index + 100),
                'user_agent' => $userAgents[$index % count($userAgents)],
                'created_at' => now()->subDays(rand(1, 30))->subHours(rand(1, 23)),
            ]);
        }

        $feedbackSubmissions = [
            [
                'name' => 'Alice Thompson',
                'email' => 'alice.t@example.com',
                'overall_rating' => '5',
                'recommend' => 'definitely',
                'improvements' => ['website'],
                'comments' => 'Excellent service! The website could be a bit faster but overall very satisfied.',
            ],
            [
                'name' => 'Robert Martinez',
                'email' => 'r.martinez@example.com',
                'overall_rating' => '4',
                'recommend' => 'probably',
                'improvements' => ['pricing', 'delivery'],
                'comments' => 'Good products but delivery took longer than expected.',
            ],
            [
                'name' => 'Jennifer Lee',
                'email' => 'jlee@example.com',
                'overall_rating' => '3',
                'recommend' => 'not_sure',
                'improvements' => ['customer_service', 'product_quality'],
                'comments' => 'The product was okay but customer service was slow to respond to my inquiries.',
            ],
        ];

        foreach ($feedbackSubmissions as $index => $submission) {
            FormBuilderSubmission::query()->create([
                'form_id' => $feedbackForm->id,
                'content' => $submission,
                'unique_identifier' => null,
                'ip_address' => '10.0.0.' . ($index + 50),
                'user_agent' => $userAgents[$index % count($userAgents)],
                'created_at' => now()->subDays(rand(1, 20))->subHours(rand(1, 23)),
            ]);
        }

        $jobSubmissions = [
            [
                'full_name' => 'Christopher Anderson',
                'email' => 'c.anderson@example.com',
                'phone' => '+1 555-111-2222',
                'linkedin' => 'https://linkedin.com/in/canderson',
                'position' => 'developer',
                'experience' => '3-5',
                'start_date' => now()->addWeeks(2)->format('Y-m-d'),
                'salary_expectation' => '$80,000 - $95,000',
                'cover_letter' => 'I am an experienced software developer with 4 years of experience in web development. I am proficient in PHP, Laravel, JavaScript, and Vue.js.',
            ],
            [
                'full_name' => 'Amanda White',
                'email' => 'awhite@example.com',
                'phone' => '+1 555-333-4444',
                'linkedin' => 'https://linkedin.com/in/amandawhite',
                'position' => 'designer',
                'experience' => '5-10',
                'start_date' => now()->addMonth()->format('Y-m-d'),
                'salary_expectation' => '$90,000 - $110,000',
                'cover_letter' => 'Creative UI/UX designer with 7 years of experience designing beautiful and user-friendly interfaces for web and mobile applications.',
            ],
            [
                'full_name' => 'Daniel Kim',
                'email' => 'dkim@example.com',
                'phone' => '+1 555-555-6666',
                'linkedin' => '',
                'position' => 'manager',
                'experience' => '10+',
                'start_date' => now()->addWeeks(4)->format('Y-m-d'),
                'salary_expectation' => '$120,000+',
                'cover_letter' => 'Seasoned project manager with over 12 years of experience leading cross-functional teams and delivering complex software projects on time and within budget.',
            ],
        ];

        foreach ($jobSubmissions as $index => $submission) {
            FormBuilderSubmission::query()->create([
                'form_id' => $jobForm->id,
                'content' => $submission,
                'unique_identifier' => null,
                'ip_address' => '172.16.0.' . ($index + 10),
                'user_agent' => $userAgents[$index % count($userAgents)],
                'created_at' => now()->subDays(rand(1, 15))->subHours(rand(1, 23)),
            ]);
        }

        $newsletterSubmissions = [
            ['email' => 'subscriber1@example.com'],
            ['email' => 'subscriber2@example.com'],
            ['email' => 'subscriber3@example.com'],
            ['email' => 'subscriber4@example.com'],
            ['email' => 'newsletter.fan@example.com'],
            ['email' => 'tech.lover@example.com'],
        ];

        foreach ($newsletterSubmissions as $index => $submission) {
            FormBuilderSubmission::query()->create([
                'form_id' => $newsletterForm->id,
                'content' => $submission,
                'unique_identifier' => $submission['email'],
                'ip_address' => '203.0.113.' . ($index + 1),
                'user_agent' => $userAgents[$index % count($userAgents)],
                'created_at' => now()->subDays(rand(1, 60))->subHours(rand(1, 23)),
            ]);
        }

        $eventSubmissions = [
            [
                'first_name' => 'Jessica',
                'last_name' => 'Robinson',
                'email' => 'jrobinson@example.com',
                'company' => 'Tech Solutions Inc.',
                'ticket_type' => 'vip',
                'attendees' => '2',
                'dietary' => 'vegetarian',
                'special_requests' => 'Would like to sit near the front if possible.',
            ],
            [
                'first_name' => 'Brian',
                'last_name' => 'Clark',
                'email' => 'bclark@example.com',
                'company' => 'Digital Marketing Agency',
                'ticket_type' => 'group',
                'attendees' => '5',
                'dietary' => 'none',
                'special_requests' => 'We would like our team to be seated together.',
            ],
            [
                'first_name' => 'Michelle',
                'last_name' => 'Lewis',
                'email' => 'mlewis@example.com',
                'company' => '',
                'ticket_type' => 'standard',
                'attendees' => '1',
                'dietary' => 'gluten_free',
                'special_requests' => '',
            ],
            [
                'first_name' => 'Steven',
                'last_name' => 'Walker',
                'email' => 'swalker@example.com',
                'company' => 'Walker Consulting',
                'ticket_type' => 'vip',
                'attendees' => '3',
                'dietary' => 'vegan',
                'special_requests' => 'Wheelchair accessible seating required for one attendee.',
            ],
        ];

        foreach ($eventSubmissions as $index => $submission) {
            FormBuilderSubmission::query()->create([
                'form_id' => $eventForm->id,
                'content' => $submission,
                'unique_identifier' => $submission['email'],
                'ip_address' => '198.51.100.' . ($index + 20),
                'user_agent' => $userAgents[$index % count($userAgents)],
                'created_at' => now()->subDays(rand(1, 10))->subHours(rand(1, 23)),
            ]);
        }
    }

    protected function truncateTables(): void
    {
        $driver = DB::getDriverName();

        if ($driver === 'mysql') {
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        }

        FormBuilderSubmission::query()->truncate();
        FormBuilder::query()->truncate();

        if ($driver === 'mysql') {
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        }
    }
}
