<?php

namespace Botble\BbFormBuilder\Classes;

use Botble\BbFormBuilder\Classes\Integrations\GetResponseIntegration;
use Botble\BbFormBuilder\Classes\Integrations\MailchimpIntegration;
use Botble\BbFormBuilder\Events\FormBuilderSubmittedEvent;
use Botble\BbFormBuilder\Exceptions\DuplicateSubmissionException;
use Botble\BbFormBuilder\Models\FormBuilder;
use Botble\BbFormBuilder\Models\FormBuilderSubmission;
use Botble\Captcha\Facades\Captcha;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class SubmissionHandler
{
    protected array $attachments = [];

    protected ?FormBuilderSubmission $submission = null;

    protected FileUploadValidator $fileValidator;

    protected array $urlTracking = [];

    public function __construct()
    {
        $this->fileValidator = new FileUploadValidator();
    }

    public function submit(Request $request, FormBuilder $form)
    {
        try {
            $this->validateRequest($request, $form);
            $this->validateFileUploads($request, $form);

            $this->urlTracking = $this->extractUrlTracking($request);

            $data = $request->except(['_token', 'g-recaptcha-response', '_fb_page_url', '_fb_referrer_url']);
            $data = $this->cleanupPhoneFields($data, $form);

            $actions = $form->actions ?? [];

            ksort($actions);

            foreach ($actions as $actionType => $actionInstances) {
                foreach ($actionInstances as $actionConfig) {
                    $this->handleAction($actionType, $actionConfig, $data, $request, $form);
                }
            }

            if ($this->submission) {
                event(new FormBuilderSubmittedEvent($this->submission));
            }

            return $this->handleSuccess($form);
        } catch (ValidationException $exception) {
            return $this->handleFailure($form, implode('<br>', $exception->validator->errors()->all()));
        } catch (Exception $exception) {
            report($exception);

            return $this->handleFailure($form, $exception->getMessage());
        }
    }

    protected function validateRequest(Request $request, FormBuilder $form): void
    {
        $rules = [];

        $maxFileSize = setting('fb_form_builder_max_file_size', config('plugins.bb-form-builder.form-builder.max_file_size', 10));

        $formFields = [];
        foreach ($form->getAllFields() as $field) {
            $formFields[$field['name'] ?? ''] = $field['type'] ?? 'text';
        }

        foreach ($request->all() as $key => $value) {
            if ($request->hasFile($key)) {
                $fieldType = $formFields[$key] ?? 'file';
                $fileKey = is_array($value) ? "{$key}.*" : $key;

                if ($fieldType === 'image') {
                    $allowedMimes = config('plugins.bb-form-builder.form-builder.image_mimes', 'jpg,jpeg,png,gif,webp');
                } else {
                    $allowedMimes = setting('fb_form_builder_allowed_mimes', config('plugins.bb-form-builder.form-builder.mimes'));
                }

                $rules[$fileKey] = "mimes:{$allowedMimes}|max:" . ($maxFileSize * 1024);
            }
        }

        if ($form->has_captcha && is_plugin_active('captcha') && Captcha::reCaptchaEnabled()) {
            $rules['g-recaptcha-response'] = 'required|captcha';
        }

        if ($form->has_math_captcha && is_plugin_active('captcha') && Captcha::mathCaptchaEnabled()) {
            $rules = [...$rules, ...Captcha::mathCaptchaRules()];
        }

        $validator = Validator::make($request->all(), $rules, [], [
            'g-recaptcha-response' => 'reCaptcha',
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
    }

    protected function validateFileUploads(Request $request, FormBuilder $form): void
    {
        $files = $request->allFiles();

        if (empty($files)) {
            return;
        }

        $formFields = [];
        foreach ($form->getAllFields() as $field) {
            $formFields[$field['name'] ?? ''] = $field['type'] ?? 'file';
        }

        foreach ($files as $fieldName => $uploadedFiles) {
            $uploadedFiles = Arr::wrap($uploadedFiles);
            $fieldType = $formFields[$fieldName] ?? 'file';

            foreach ($uploadedFiles as $file) {
                if (! $file instanceof UploadedFile || ! $file->isValid()) {
                    continue;
                }

                if (! $this->fileValidator->validate($file, $fieldType)) {
                    $error = $this->fileValidator->getFirstError()
                        ?? trans('plugins/bb-form-builder::form.validation.invalid_file');

                    throw ValidationException::withMessages([$fieldName => $error]);
                }
            }
        }
    }

    protected function handleAction(string $actionType, array $actionConfig, array $data, Request $request, FormBuilder $form): void
    {
        $handlerMethod = "{$actionType}ActionHandler";

        if (method_exists($this, $handlerMethod)) {
            $this->{$handlerMethod}($actionConfig, $data, $request, $form);
        }
    }

    protected function emailActionHandler(array $action, array $data, Request $request, FormBuilder $form): void
    {
        $to = array_map('trim', explode(',', Arr::get($action, 'to', '')));
        $subject = Arr::get($action, 'subject', 'Form Submission');
        $body = Arr::get($action, 'body', '');

        if (empty($to) || empty($subject)) {
            return;
        }

        $body = $this->replaceFieldsInContent($body, $data);
        $subject = $this->replaceFieldsInContent($subject, $data);

        $urlTrackingHtml = $this->buildUrlTrackingHtml();
        if ($urlTrackingHtml) {
            $body .= "\n\n" . $urlTrackingHtml;
        }

        $attachments = $this->processFileUploads($request, $form);

        Mail::send([], [], function ($message) use ($to, $subject, $body, $attachments, $form): void {
            $message->to($to)
                ->subject($subject)
                ->html(nl2br(e($body)));

            foreach ($attachments as $attachment) {
                if (file_exists($attachment)) {
                    $message->attach($attachment);
                }
            }
        });
    }

    protected function databaseActionHandler(array $action, array $data, Request $request, FormBuilder $form): void
    {
        $uniqueField = Arr::get($action, 'unique_field');
        $submissionData = [
            'unique_identifier' => null,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'page_url' => $this->urlTracking['page_url'] ?? null,
            'referrer_url' => $this->urlTracking['referrer_url'] ?? null,
        ];

        if (! empty($uniqueField)) {
            $uniqueValue = Arr::get($data, $uniqueField);

            if ($uniqueValue) {
                $existingSubmission = $form->submissions()
                    ->where('unique_identifier', $uniqueValue)
                    ->first();

                if ($existingSubmission) {
                    throw new DuplicateSubmissionException(trans('plugins/bb-form-builder::form.messages.duplicate_submission'));
                }

                $submissionData['unique_identifier'] = $uniqueValue;
            }
        }

        $submissionData['content'] = $data;

        $this->submission = $form->submissions()->create($submissionData);

        $this->processFileUploadsForSubmission($request, $form, $this->submission);
    }

    protected function apiActionHandler(array $action, array $data, Request $request, FormBuilder $form): void
    {
        $endpoint = Arr::get($action, 'end_point');
        $method = strtoupper(Arr::get($action, 'method', 'POST'));
        $body = Arr::get($action, 'body', '{}');

        if (empty($endpoint)) {
            return;
        }

        $body = $this->replaceFieldsInContent($body, $data);
        $parsedBody = json_decode($body, true) ?? [];

        try {
            if ($method === 'GET') {
                Http::timeout(30)->get($endpoint, $parsedBody);
            } else {
                Http::timeout(30)->post($endpoint, $parsedBody);
            }
        } catch (Exception $e) {
            report($e);
        }
    }

    protected function mailchimpActionHandler(array $action, array $data, Request $request, FormBuilder $form): void
    {
        $listId = Arr::get($action, 'list_id');
        $emailField = Arr::get($action, 'email_field');
        $nameField = Arr::get($action, 'name_field');

        if (empty($listId) || empty($emailField)) {
            return;
        }

        $email = Arr::get($data, $emailField);

        if (empty($email) || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return;
        }

        $name = $nameField ? Arr::get($data, $nameField) : null;

        $apiKey = setting('fb_form_builder_mailchimp_api_key');

        if (empty($apiKey)) {
            Log::warning('Mailchimp action failed: API key not configured');

            return;
        }

        try {
            $integration = new MailchimpIntegration($apiKey);
            $integration->subscribe($listId, $email, $name);
        } catch (Exception $e) {
            Log::error('Mailchimp subscription failed', [
                'email' => $email,
                'error' => $e->getMessage(),
            ]);
        }
    }

    protected function getresponseActionHandler(array $action, array $data, Request $request, FormBuilder $form): void
    {
        $campaignId = Arr::get($action, 'campaign_id');
        $emailField = Arr::get($action, 'email_field');
        $nameField = Arr::get($action, 'name_field');

        if (empty($campaignId) || empty($emailField)) {
            return;
        }

        $email = Arr::get($data, $emailField);

        if (empty($email) || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return;
        }

        $name = $nameField ? Arr::get($data, $nameField) : null;

        $apiKey = setting('fb_form_builder_getresponse_api_key');

        if (empty($apiKey)) {
            Log::warning('GetResponse action failed: API key not configured');

            return;
        }

        try {
            $integration = new GetResponseIntegration($apiKey);
            $integration->subscribe($campaignId, $email, $name);
        } catch (Exception $e) {
            Log::error('GetResponse subscription failed', [
                'email' => $email,
                'error' => $e->getMessage(),
            ]);
        }
    }

    protected function cleanupPhoneFields(array $data, FormBuilder $form): array
    {
        $phoneFields = [];

        foreach ($form->getAllFields() as $field) {
            if (($field['type'] ?? '') === 'phone' && ($field['enable_country_code'] ?? true)) {
                $phoneFields[] = $field['name'] ?? '';
            }
        }

        foreach ($phoneFields as $fieldName) {
            $displayKey = $fieldName . '_display';

            if (isset($data[$displayKey])) {
                if (empty($data[$fieldName]) && ! empty($data[$displayKey])) {
                    $data[$fieldName] = $data[$displayKey];
                }

                unset($data[$displayKey]);
            }
        }

        return $data;
    }

    protected function replaceFieldsInContent(string $content, array $data): string
    {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $value = implode(', ', $value);
            }

            $content = preg_replace('/\[' . preg_quote($key, '/') . '\]/', (string) $value, $content);
        }

        return $content;
    }

    protected function processFileUploads(Request $request, FormBuilder $form): array
    {
        if (! empty($this->attachments)) {
            return $this->attachments;
        }

        $attachments = [];
        $files = $request->allFiles();
        $disk = Storage::disk('local');

        foreach ($files as $fieldName => $uploadedFiles) {
            $uploadedFiles = Arr::wrap($uploadedFiles);

            foreach ($uploadedFiles as $file) {
                $sanitizedName = $this->fileValidator->sanitizeFilename($file->getClientOriginalName());
                $filename = time() . '-' . $sanitizedName;
                $storagePath = 'form-builder/' . $form->id . '/' . $filename;

                $disk->putFileAs('form-builder/' . $form->id, $file, $filename);
                $attachments[] = $disk->path($storagePath);
            }
        }

        $this->attachments = $attachments;

        return $attachments;
    }

    protected function processFileUploadsForSubmission(Request $request, FormBuilder $form, FormBuilderSubmission $submission): void
    {
        $files = $request->allFiles();

        if (empty($files)) {
            return;
        }

        $content = $submission->content;
        $disk = Storage::disk('local');

        foreach ($files as $fieldName => $uploadedFiles) {
            $uploadedFiles = Arr::wrap($uploadedFiles);
            $fileLinks = [];

            foreach ($uploadedFiles as $file) {
                $originalName = $file->getClientOriginalName();
                $sanitizedName = $this->fileValidator->sanitizeFilename($originalName);
                $filename = time() . '-' . $sanitizedName;
                $storagePath = 'form-builder/' . $submission->id . '/' . $filename;

                $disk->putFileAs('form-builder/' . $submission->id, $file, $filename);

                $this->attachments[] = $disk->path($storagePath);

                $downloadUrl = route('bb-form-builder.submissions.download', [
                    'submission' => $submission->id,
                    'filename' => $filename,
                ]);

                $fileLinks[] = '<a href="' . $downloadUrl . '" target="_blank">' . e($originalName) . '</a>';
            }

            $content[$fieldName] = implode(' | ', $fileLinks);
        }

        $submission->content = $content;
        $submission->save();
    }

    protected function handleSuccess(FormBuilder $form): JsonResponse
    {
        $submission = $form->getTranslatedSubmission();
        $onSuccess = Arr::get($submission, 'on_success', []);
        $action = Arr::get($onSuccess, 'action', 'show_message');
        $content = Arr::get($onSuccess, 'content', trans('plugins/bb-form-builder::form.messages.submission_success'));

        if ($action === 'redirect_to' && ! empty($content)) {
            return response()->json([
                'redirect' => $content,
            ]);
        }

        return response()->json([
            'message' => $content,
            'success' => true,
        ]);
    }

    protected function handleFailure(FormBuilder $form, string $errorMessage = ''): JsonResponse
    {
        $submission = $form->getTranslatedSubmission();
        $onFailure = Arr::get($submission, 'on_failure', []);
        $action = Arr::get($onFailure, 'action', 'show_message');
        $content = Arr::get($onFailure, 'content', trans('plugins/bb-form-builder::form.messages.submission_failed'));

        if ($action === 'redirect_to' && ! empty($content)) {
            return response()->json([
                'redirect' => $content,
            ], 422);
        }

        if (! empty($errorMessage)) {
            $content .= '<br>' . $errorMessage;
        }

        return response()->json([
            'message' => $content,
            'success' => false,
        ], 422);
    }

    protected function extractUrlTracking(Request $request): array
    {
        $pageUrl = $request->input('_fb_page_url');
        $referrerUrl = $request->input('_fb_referrer_url');

        if (empty($pageUrl)) {
            $pageUrl = $request->headers->get('referer');
        }

        return [
            'page_url' => $pageUrl ? $this->sanitizeUrl($pageUrl) : null,
            'referrer_url' => $referrerUrl ? $this->sanitizeUrl($referrerUrl) : null,
        ];
    }

    protected function sanitizeUrl(?string $url): ?string
    {
        if (empty($url)) {
            return null;
        }

        $url = filter_var($url, FILTER_SANITIZE_URL);

        if (! filter_var($url, FILTER_VALIDATE_URL)) {
            return null;
        }

        if (strlen($url) > 2048) {
            $url = substr($url, 0, 2048);
        }

        return $url;
    }

    protected function buildUrlTrackingHtml(): string
    {
        $lines = [];

        $lines[] = '---';
        $lines[] = trans('plugins/bb-form-builder::form.submissions.url_tracking_title');

        if (! empty($this->urlTracking['page_url'])) {
            $lines[] = trans('plugins/bb-form-builder::form.submissions.page_url') . ': ' . $this->urlTracking['page_url'];
        }

        if (! empty($this->urlTracking['referrer_url'])) {
            $lines[] = trans('plugins/bb-form-builder::form.submissions.referrer_url') . ': ' . $this->urlTracking['referrer_url'];
        }

        if (count($lines) <= 2) {
            return '';
        }

        return implode("\n", $lines);
    }
}
