<?php

namespace Botble\BbFormBuilder\Classes\Integrations;

use Illuminate\Support\Facades\Log;

class MailchimpIntegration extends BaseIntegration
{
    protected string $dataCenter;

    public function __construct(string $apiKey)
    {
        parent::__construct($apiKey);

        $this->dataCenter = $this->extractDataCenter($apiKey);
        $this->baseUrl = "https://{$this->dataCenter}.api.mailchimp.com/3.0";
    }

    protected function extractDataCenter(string $apiKey): string
    {
        if (str_contains($apiKey, '-')) {
            return substr($apiKey, strpos($apiKey, '-') + 1);
        }

        return 'us1';
    }

    public function getLists(): array
    {
        if (! $this->isConfigured()) {
            return [];
        }

        $response = $this->get('/lists?count=100');

        if (empty($response['lists'])) {
            return [];
        }

        return collect($response['lists'])
            ->map(fn ($list) => [
                'id' => $list['id'],
                'name' => $list['name'],
                'member_count' => $list['stats']['member_count'] ?? 0,
            ])
            ->toArray();
    }

    public function subscribe(string $listId, string $email, ?string $name = null): bool
    {
        if (! $this->isConfigured()) {
            return false;
        }

        $data = [
            'email_address' => $email,
            'status' => 'subscribed',
        ];

        if ($name) {
            $nameParts = explode(' ', $name, 2);
            $data['merge_fields'] = [
                'FNAME' => $nameParts[0] ?? '',
                'LNAME' => $nameParts[1] ?? '',
            ];
        }

        $response = $this->post("/lists/{$listId}/members", $data);

        if (isset($response['error'])) {
            if ($response['status'] === 400) {
                Log::info('Mailchimp: Member already exists or invalid email', ['email' => $email]);

                return true;
            }

            return false;
        }

        return isset($response['id']);
    }

    protected function getHeaders(array $additional = []): array
    {
        return array_merge(parent::getHeaders($additional), [
            'Authorization' => 'Basic ' . base64_encode('anystring:' . $this->apiKey),
        ]);
    }
}
