<?php

namespace Botble\BbFormBuilder\Classes\Integrations;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

abstract class BaseIntegration
{
    protected string $apiKey;

    protected string $baseUrl;

    public function __construct(string $apiKey)
    {
        $this->apiKey = $apiKey;
    }

    abstract public function getLists(): array;

    abstract public function subscribe(string $listId, string $email, ?string $name = null): bool;

    protected function get(string $endpoint, array $headers = []): array
    {
        try {
            $response = Http::withHeaders($this->getHeaders($headers))
                ->timeout(30)
                ->get($this->baseUrl . $endpoint);

            if ($response->successful()) {
                return $response->json() ?? [];
            }

            Log::error(static::class . ' API GET error', [
                'endpoint' => $endpoint,
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            return [];
        } catch (\Exception $e) {
            Log::error(static::class . ' API GET exception', [
                'endpoint' => $endpoint,
                'message' => $e->getMessage(),
            ]);

            return [];
        }
    }

    protected function post(string $endpoint, array $data, array $headers = []): array
    {
        try {
            $response = Http::withHeaders($this->getHeaders($headers))
                ->timeout(30)
                ->post($this->baseUrl . $endpoint, $data);

            if ($response->successful()) {
                return $response->json() ?? [];
            }

            Log::error(static::class . ' API POST error', [
                'endpoint' => $endpoint,
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            return ['error' => true, 'status' => $response->status()];
        } catch (\Exception $e) {
            Log::error(static::class . ' API POST exception', [
                'endpoint' => $endpoint,
                'message' => $e->getMessage(),
            ]);

            return ['error' => true, 'message' => $e->getMessage()];
        }
    }

    protected function getHeaders(array $additional = []): array
    {
        return array_merge([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ], $additional);
    }

    public function isConfigured(): bool
    {
        return ! empty($this->apiKey);
    }
}
