<?php

namespace Botble\BbFormBuilder\Classes;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Str;

class FileUploadValidator
{
    protected array $dangerousExtensions;

    protected array $mimeTypeMapping;

    protected array $errors = [];

    public function __construct()
    {
        $this->dangerousExtensions = config('plugins.bb-form-builder.form-builder.dangerous_extensions', []);
        $this->mimeTypeMapping = config('plugins.bb-form-builder.form-builder.mime_type_mapping', []);
    }

    public function validate(UploadedFile $file, string $fieldType = 'file'): bool
    {
        $this->errors = [];

        $originalName = $file->getClientOriginalName();

        if (! $this->validateExtension($originalName)) {
            return false;
        }

        if (! $this->validateMimeType($file, $fieldType)) {
            return false;
        }

        if (! $this->validateFileContent($file)) {
            return false;
        }

        if ($fieldType === 'image' && ! $this->validateImageFile($file)) {
            return false;
        }

        return true;
    }

    protected function validateExtension(string $filename): bool
    {
        $parts = explode('.', strtolower($filename));
        array_shift($parts);

        foreach ($parts as $extension) {
            if (in_array($extension, $this->dangerousExtensions, true)) {
                $this->errors[] = trans('plugins/bb-form-builder::form.validation.dangerous_extension', [
                    'extension' => $extension,
                ]);

                return false;
            }
        }

        if (str_contains($filename, "\0")) {
            $this->errors[] = trans('plugins/bb-form-builder::form.validation.invalid_filename');

            return false;
        }

        return true;
    }

    protected function validateMimeType(UploadedFile $file, string $fieldType): bool
    {
        $extension = strtolower($file->getClientOriginalExtension());
        $actualMimeType = $file->getMimeType();

        if ($fieldType === 'image') {
            $allowedImageMimes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];

            if (! in_array($actualMimeType, $allowedImageMimes, true)) {
                $this->errors[] = trans('plugins/bb-form-builder::form.validation.invalid_image_type');

                return false;
            }
        }

        if (isset($this->mimeTypeMapping[$extension])) {
            $expectedMimes = $this->mimeTypeMapping[$extension];

            if (! in_array($actualMimeType, $expectedMimes, true)) {
                $this->errors[] = trans('plugins/bb-form-builder::form.validation.mime_mismatch');

                return false;
            }
        }

        return true;
    }

    protected function validateFileContent(UploadedFile $file): bool
    {
        $content = file_get_contents($file->getRealPath(), false, null, 0, 8192);

        if ($content === false) {
            return true;
        }

        $phpPatterns = [
            '<?php',
            '<?=',
            '<? ',
            '<%',
            '<script language="php"',
            '<script language=\'php\'',
        ];

        foreach ($phpPatterns as $pattern) {
            if (stripos($content, $pattern) !== false) {
                $this->errors[] = trans('plugins/bb-form-builder::form.validation.malicious_content');

                return false;
            }
        }

        $shellPatterns = [
            '#!/bin/bash',
            '#!/bin/sh',
            '#!/usr/bin/env',
            '#!/usr/bin/perl',
            '#!/usr/bin/python',
        ];

        foreach ($shellPatterns as $pattern) {
            if (stripos($content, $pattern) !== false) {
                $this->errors[] = trans('plugins/bb-form-builder::form.validation.malicious_content');

                return false;
            }
        }

        $webShellPatterns = [
            'eval(base64_decode',
            'eval(gzinflate',
            'eval(gzuncompress',
            'eval(str_rot13',
            'base64_decode(gzinflate',
            'assert(base64_decode',
            'preg_replace.*\/e',
            'create_function',
            'call_user_func',
            'passthru',
            'shell_exec',
            'system(',
            'exec(',
            'popen(',
            'proc_open',
        ];

        foreach ($webShellPatterns as $pattern) {
            if (preg_match('/' . preg_quote($pattern, '/') . '/i', $content)) {
                $this->errors[] = trans('plugins/bb-form-builder::form.validation.malicious_content');

                return false;
            }
        }

        return true;
    }

    protected function validateImageFile(UploadedFile $file): bool
    {
        $imageInfo = @getimagesize($file->getRealPath());

        if ($imageInfo === false) {
            $this->errors[] = trans('plugins/bb-form-builder::form.validation.invalid_image');

            return false;
        }

        $validImageTypes = [
            IMAGETYPE_JPEG,
            IMAGETYPE_PNG,
            IMAGETYPE_GIF,
            IMAGETYPE_WEBP,
        ];

        if (! in_array($imageInfo[2], $validImageTypes, true)) {
            $this->errors[] = trans('plugins/bb-form-builder::form.validation.invalid_image_type');

            return false;
        }

        return true;
    }

    public function sanitizeFilename(string $filename): string
    {
        $extension = pathinfo($filename, PATHINFO_EXTENSION);
        $name = pathinfo($filename, PATHINFO_FILENAME);

        $name = basename($name);

        $name = preg_replace('/[^a-zA-Z0-9_\-]/', '_', $name);

        $name = preg_replace('/_+/', '_', $name);

        $name = trim($name, '_');

        if (empty($name)) {
            $name = 'file_' . Str::random(8);
        }

        $extension = preg_replace('/[^a-zA-Z0-9]/', '', $extension);

        return $name . '.' . strtolower($extension);
    }

    public function getErrors(): array
    {
        return $this->errors;
    }

    public function getFirstError(): ?string
    {
        return $this->errors[0] ?? null;
    }
}
