(function ($) {
    'use strict';

    if (typeof $ !== 'function') {
        console.warn('BB Form Builder: jQuery is required for public scripts.');
        return;
    }

    var FbFormPublic = {
        storagePrefix: 'fb_form_',

        init: function () {
            this.initForms();
            this.initWizards();
            this.initStarRatings();
            this.initFileUploads();
            this.initFormPersistence();
            this.initAutocomplete();
            this.initPopups();
            this.initUrlTracking();
        },

        getStorageKey: function (formHash) {
            return this.storagePrefix + formHash;
        },

        saveFormData: function ($form) {
            var formHash = $form.closest('.fb-form-container').data('form-hash');
            if (!formHash) return;

            var formData = {};
            var $wizard = $form.find('.fb-form-wizard');

            if ($wizard.length && $wizard.data('fobWizard')) {
                formData._currentStep = $wizard.data('fobWizard').currentStep();
            }

            $form.find('input, select, textarea').each(function () {
                var $field = $(this);
                var name = $field.attr('name');
                var type = $field.attr('type');

                if (!name || type === 'file' || name === '_token') return;

                if (type === 'checkbox') {
                    if (!formData[name]) {
                        formData[name] = [];
                    }
                    if ($field.is(':checked')) {
                        formData[name].push($field.val());
                    }
                } else if (type === 'radio') {
                    if ($field.is(':checked')) {
                        formData[name] = $field.val();
                    }
                } else {
                    formData[name] = $field.val();
                }
            });

            try {
                localStorage.setItem(this.getStorageKey(formHash), JSON.stringify(formData));
            } catch (e) {
                console.warn('Could not save form data to localStorage:', e);
            }
        },

        restoreFormData: function ($form) {
            var self = this;
            var formHash = $form.closest('.fb-form-container').data('form-hash');
            if (!formHash) return;

            try {
                var savedData = localStorage.getItem(this.getStorageKey(formHash));
                if (!savedData) return;

                var formData = JSON.parse(savedData);
                var hasData = false;

                Object.keys(formData).forEach(function (name) {
                    if (name === '_currentStep') return;

                    var value = formData[name];
                    var $field = $form.find('[name="' + name + '"]');

                    if (!$field.length) return;

                    var type = $field.attr('type');

                    if (type === 'checkbox') {
                        $field.each(function () {
                            var $cb = $(this);
                            if (Array.isArray(value) && value.includes($cb.val())) {
                                $cb.prop('checked', true);
                                hasData = true;
                            }
                        });
                    } else if (type === 'radio') {
                        $field.filter('[value="' + value + '"]').prop('checked', true);
                        if (value) hasData = true;
                    } else if ($field.is('select')) {
                        $field.val(value);
                        if (value) hasData = true;
                    } else {
                        $field.val(value);
                        if (value) hasData = true;
                    }
                });

                $form.find('.fb-star-rating').each(function () {
                    var $container = $(this);
                    var $input = $container.find('input[type="hidden"]');
                    var name = $input.attr('name');
                    var value = formData[name];

                    if (value && typeof $.fn.rateYo !== 'undefined') {
                        var $ratingEl = $container.find('[data-rateyo]');
                        if ($ratingEl.data('rateYo')) {
                            $ratingEl.rateYo('rating', parseFloat(value));
                        }
                    }
                });

                var $wizard = $form.find('.fb-form-wizard');
                if ($wizard.length && formData._currentStep !== undefined && hasData) {
                    setTimeout(function () {
                        if ($wizard.data('fobWizard')) {
                            $wizard.data('fobWizard').goToStep(formData._currentStep);
                        }
                    }, 100);
                }

                if (hasData) {
                    self.showRestoreNotification($form, formHash);
                }
            } catch (e) {
                console.warn('Could not restore form data from localStorage:', e);
            }
        },

        showRestoreNotification: function ($form, formHash) {
            var self = this;
            var $wrapper = $form.closest('.fb-form-container');
            var progressRestoredMsg = $form.data('msg-progress-restored') || 'Your previous progress has been restored.';
            var startFreshMsg = $form.data('msg-start-fresh') || 'Start Fresh';

            var $notification = $('<div class="fb-restore-notification">' +
                '<span>' + progressRestoredMsg + '</span>' +
                '<button type="button" class="fb-restore-clear">' + startFreshMsg + '</button>' +
                '</div>');

            var $verticalMain = $wrapper.find('.fb-wizard-vertical-container .fb-wizard-main');
            if ($verticalMain.length) {
                $verticalMain.prepend($notification);
            } else {
                $wrapper.find('.fb-form').prepend($notification);
            }

            $notification.find('.fb-restore-clear').on('click', function () {
                self.clearFormData(formHash);
                $form[0].reset();

                var $wizard = $form.find('.fb-form-wizard');
                if ($wizard.length && $wizard.data('fobWizard')) {
                    $wizard.data('fobWizard').reset();
                }

                $form.find('.fb-star-rating').each(function () {
                    var $ratingEl = $(this).find('[data-rateyo]');
                    if ($ratingEl.data('rateYo')) {
                        $ratingEl.rateYo('rating', 0);
                    }
                });

                $notification.slideUp(300, function () {
                    $(this).remove();
                });
            });

            setTimeout(function () {
                $notification.slideUp(300, function () {
                    $(this).remove();
                });
            }, 10000);
        },

        clearFormData: function (formHash) {
            try {
                localStorage.removeItem(this.getStorageKey(formHash));
            } catch (e) {
                console.warn('Could not clear form data from localStorage:', e);
            }
        },

        initFormPersistence: function () {
            var self = this;
            var isEmbedded = self.isInCrossOriginIframe();

            $('.fb-form').each(function () {
                var $form = $(this);
                var disablePersistence = $form.data('disable-persistence') === true || isEmbedded;

                if (!disablePersistence) {
                    self.restoreFormData($form);
                }
            });

            var saveTimeout;
            $(document).on('change input', '.fb-form input, .fb-form select, .fb-form textarea', function () {
                var $form = $(this).closest('.fb-form');

                if ($form.data('disable-persistence') === true || isEmbedded) {
                    return;
                }

                clearTimeout(saveTimeout);
                saveTimeout = setTimeout(function () {
                    self.saveFormData($form);
                }, 300);
            });
        },

        isInCrossOriginIframe: function () {
            try {
                if (window.self === window.top) {
                    return false;
                }

                var test = window.parent.location.href;
                return false;
            } catch (e) {
                return true;
            }
        },

        initForms: function () {
            var self = this;

            $(document).on('submit', '.fb-form', function (e) {
                e.preventDefault();
                self.submitForm($(this));
            });
        },

        initWizards: function () {
            var self = this;
            var $wizards = $('.fb-form-wizard');

            $wizards.each(function () {
                var $wizard = $(this);
                var $form = $wizard.closest('form');
                var $steps = $wizard.find('.fb-wizard-step');
                var $panels = $wizard.find('.fb-wizard-panel');
                var $prevBtn = $wizard.find('.fb-wizard-prev');
                var $nextBtn = $wizard.find('.fb-wizard-next');
                var $submitBtn = $wizard.find('.fb-wizard-submit');
                var currentStep = 0;
                var totalSteps = $steps.length;

                $panels.addClass('step-content');

                var stepConditions = {};
                $panels.each(function (index) {
                    var $panel = $(this);
                    var stepId = $panel.attr('id');
                    if (stepId) {
                        stepId = stepId.replace('fb-panel-', '');
                    }
                    var conditionField = $panel.data('condition-field');
                    var conditionOperator = $panel.data('condition-operator');
                    var conditionValue = $panel.data('condition-value');
                    if (conditionField) {
                        stepConditions[index] = {
                            field: conditionField,
                            operator: conditionOperator || 'checked',
                            value: conditionValue || ''
                        };
                    }
                });

                function isStepConditionMet(stepIndex) {
                    var condition = stepConditions[stepIndex];
                    if (!condition || !condition.field) {
                        return true;
                    }

                    var fieldName = condition.field;
                    var operator = condition.operator;
                    var expectedValue = condition.value;
                    var $field = $form.find('[name="' + fieldName + '"], [name="' + fieldName + '[]"], [name^="' + fieldName + '["]');

                    if (!$field.length) {
                        return true;
                    }

                    var fieldType = $field.attr('type') || $field.prop('tagName').toLowerCase();
                    var fieldValue = '';
                    var isChecked = false;

                    if (fieldType === 'checkbox') {
                        if ($field.length === 1) {
                            isChecked = $field.is(':checked');
                            fieldValue = isChecked ? $field.val() : '';
                        } else {
                            var checkedValues = [];
                            $field.filter(':checked').each(function () {
                                checkedValues.push($(this).val());
                            });
                            isChecked = checkedValues.length > 0;
                            fieldValue = checkedValues.join(',');
                        }
                    } else if (fieldType === 'radio') {
                        var $checked = $field.filter(':checked');
                        isChecked = $checked.length > 0;
                        fieldValue = $checked.val() || '';
                    } else if (fieldType === 'select') {
                        fieldValue = $field.val() || '';
                        isChecked = fieldValue !== '';
                    } else {
                        fieldValue = $field.val() || '';
                        isChecked = fieldValue !== '';
                    }

                    switch (operator) {
                        case 'checked':
                            return isChecked;
                        case 'not_checked':
                            return !isChecked;
                        case 'equals':
                            if (Array.isArray(fieldValue)) {
                                return fieldValue.includes(expectedValue);
                            }
                            if (typeof fieldValue === 'string' && fieldValue.indexOf(',') !== -1) {
                                return fieldValue.split(',').includes(expectedValue);
                            }
                            return fieldValue === expectedValue;
                        case 'not_equals':
                            if (Array.isArray(fieldValue)) {
                                return !fieldValue.includes(expectedValue);
                            }
                            if (typeof fieldValue === 'string' && fieldValue.indexOf(',') !== -1) {
                                return !fieldValue.split(',').includes(expectedValue);
                            }
                            return fieldValue !== expectedValue;
                        default:
                            return true;
                    }
                }

                function findNextVisibleStep(fromIndex) {
                    for (var i = fromIndex + 1; i < totalSteps; i++) {
                        if (isStepConditionMet(i)) {
                            return i;
                        }
                    }
                    return -1;
                }

                function findPrevVisibleStep(fromIndex) {
                    for (var i = fromIndex - 1; i >= 0; i--) {
                        if (isStepConditionMet(i)) {
                            return i;
                        }
                    }
                    return -1;
                }

                function isLastVisibleStep(index) {
                    return findNextVisibleStep(index) === -1;
                }

                function showStep(index) {
                    $steps.each(function (i) {
                        var $step = $(this);
                        $step.removeClass('active done skipped');

                        if (i < index) {
                            if (isStepConditionMet(i)) {
                                $step.addClass('done');
                            } else {
                                $step.addClass('skipped');
                            }
                        } else if (i === index) {
                            $step.addClass('active');
                        } else {
                            if (!isStepConditionMet(i)) {
                                $step.addClass('skipped');
                            }
                        }
                    });

                    $panels.removeClass('active');
                    $panels.eq(index).addClass('active');

                    var hasPrev = findPrevVisibleStep(index) >= 0;
                    var isLast = isLastVisibleStep(index);

                    $prevBtn.toggle(hasPrev);
                    $nextBtn.toggle(!isLast);
                    $submitBtn.toggle(isLast);

                    currentStep = index;

                    var wizardOffset = $wizard.offset();
                    if (wizardOffset && $(window).scrollTop() > wizardOffset.top - 20) {
                        $('html, body').animate({
                            scrollTop: wizardOffset.top - 20
                        }, 300);
                    }
                }

                showStep(0);

                $prevBtn.on('click', function () {
                    var prevStep = findPrevVisibleStep(currentStep);
                    if (prevStep >= 0) {
                        showStep(prevStep);
                    }
                });

                $nextBtn.on('click', function () {
                    var $currentPanel = $panels.eq(currentStep);
                    if (self.validateStep($currentPanel)) {
                        var nextStep = findNextVisibleStep(currentStep);
                        if (nextStep >= 0) {
                            showStep(nextStep);
                        }
                    }
                });

                $steps.on('click', function () {
                    var targetIndex = $steps.index(this);
                    var $targetStep = $(this);

                    if ($targetStep.hasClass('skipped')) {
                        return;
                    }

                    if ($targetStep.hasClass('done')) {
                        showStep(targetIndex);
                    } else if (targetIndex === findNextVisibleStep(currentStep)) {
                        var $currentPanel = $panels.eq(currentStep);
                        if (self.validateStep($currentPanel)) {
                            showStep(targetIndex);
                        }
                    }
                });

                $form.on('change', 'input, select', function () {
                    $steps.each(function (i) {
                        var $step = $(this);
                        if (i !== currentStep && !$step.hasClass('done')) {
                            if (!isStepConditionMet(i)) {
                                $step.addClass('skipped');
                            } else {
                                $step.removeClass('skipped');
                            }
                        }
                    });

                    var isLast = isLastVisibleStep(currentStep);
                    $nextBtn.toggle(!isLast);
                    $submitBtn.toggle(isLast);
                });

                $wizard.data('fobWizard', {
                    reset: function () {
                        showStep(0);
                        $steps.removeClass('done skipped');
                    },
                    currentStep: function () {
                        return currentStep;
                    },
                    goToStep: function (index) {
                        if (index >= 0 && index < totalSteps && isStepConditionMet(index)) {
                            showStep(index);
                        }
                    },
                    isStepConditionMet: isStepConditionMet,
                    findNextVisibleStep: findNextVisibleStep,
                    findPrevVisibleStep: findPrevVisibleStep
                });
            });
        },

        initStarRatings: function () {
            $('.fb-star-rating').each(function () {
                var $container = $(this);
                var $input = $container.find('input[type="hidden"]');
                var $ratingEl = $container.find('[data-rateyo]');

                if (typeof $.fn.rateYo !== 'undefined') {
                    $ratingEl.rateYo({
                        rating: $input.val() || 0,
                        fullStar: true,
                        starWidth: '24px',
                        ratedFill: '#f39c12',
                        normalFill: '#ddd',
                        onSet: function (rating, rateYoInstance) {
                            $input.val(rating);
                        }
                    });
                }
            });
        },

        initFileUploads: function () {
            var self = this;

            $('.fb-file-upload-wrapper').each(function () {
                var $wrapper = $(this);
                var $dropzone = $wrapper.find('.fb-file-dropzone');
                var $input = $wrapper.find('.fb-file-input');
                var $fileList = $wrapper.find('.fb-file-list');

                $dropzone.on('dragenter dragover', function (e) {
                    e.preventDefault();
                    e.stopPropagation();
                    $(this).addClass('dragover');
                });

                $dropzone.on('dragleave drop', function (e) {
                    e.preventDefault();
                    e.stopPropagation();
                    $(this).removeClass('dragover');
                });

                $dropzone.on('drop', function (e) {
                    var files = e.originalEvent.dataTransfer.files;
                    $input[0].files = files;
                    self.updateFileList($input, $fileList);
                });

                $input.on('change', function () {
                    self.updateFileList($input, $fileList);
                });
            });
        },

        initAutocomplete: function () {
            var self = this;

            $('.fb-autocomplete-wrapper').each(function () {
                var $wrapper = $(this);
                var $input = $wrapper.find('.fb-autocomplete-input');
                var $dropdown = $wrapper.find('.fb-autocomplete-dropdown');
                var options = [];

                try {
                    options = JSON.parse($input.data('options') || '[]');
                } catch (e) {
                    options = [];
                }

                if (options.length === 0) return;

                var selectedIndex = -1;

                function showDropdown(filter) {
                    var filtered = options.filter(function (opt) {
                        return opt.toLowerCase().indexOf(filter.toLowerCase()) !== -1;
                    });

                    if (filtered.length === 0 || filter.length === 0) {
                        $dropdown.hide();
                        return;
                    }

                    var html = '';
                    filtered.slice(0, 10).forEach(function (opt, index) {
                        html += '<div class="fb-autocomplete-item' + (index === selectedIndex ? ' selected' : '') + '" data-value="' + self.escapeHtml(opt) + '">' + self.escapeHtml(opt) + '</div>';
                    });

                    $dropdown.html(html).show();
                }

                $input.on('input', function () {
                    selectedIndex = -1;
                    showDropdown($(this).val());
                });

                $input.on('focus', function () {
                    if ($(this).val()) {
                        showDropdown($(this).val());
                    }
                });

                $input.on('blur', function () {
                    setTimeout(function () {
                        $dropdown.hide();
                    }, 200);
                });

                $input.on('keydown', function (e) {
                    var $items = $dropdown.find('.fb-autocomplete-item');

                    if (e.keyCode === 40) {
                        e.preventDefault();
                        selectedIndex = Math.min(selectedIndex + 1, $items.length - 1);
                        $items.removeClass('selected').eq(selectedIndex).addClass('selected');
                    } else if (e.keyCode === 38) {
                        e.preventDefault();
                        selectedIndex = Math.max(selectedIndex - 1, 0);
                        $items.removeClass('selected').eq(selectedIndex).addClass('selected');
                    } else if (e.keyCode === 13) {
                        e.preventDefault();
                        var $selected = $items.filter('.selected');
                        if ($selected.length) {
                            $input.val($selected.data('value'));
                            $dropdown.hide();
                        }
                    } else if (e.keyCode === 27) {
                        $dropdown.hide();
                    }
                });

                $dropdown.on('click', '.fb-autocomplete-item', function () {
                    $input.val($(this).data('value'));
                    $dropdown.hide();
                });
            });
        },

        initPopups: function () {
            var self = this;
            var $popups = $('.fb-popup-overlay');

            if ($popups.length === 0) return;

            var isAnyPopupOpen = false;

            $popups.each(function (index) {
                var $popup = $(this);
                var trigger = $popup.data('trigger');
                var popupId = $popup.attr('id');
                var closeOnOverlay = $popup.data('close-on-overlay') === 'true';
                var popupShownKey = 'fb_popup_shown_' + popupId;
                var eventNamespace = '.fbPopup' + index;

                if ($popup.data('fb-initialized')) return;
                $popup.data('fb-initialized', true);

                if (!$popup.parent().is('body')) {
                    $popup.appendTo('body');
                }

                $('[data-popup-target="' + popupId + '"]').on('click', function (e) {
                    e.preventDefault();
                    e.stopPropagation();
                    self.openPopup($popup);
                });

                $popup.find('.fb-popup-close').on('click', function (e) {
                    e.preventDefault();
                    e.stopPropagation();
                    self.closePopup($popup);
                });

                if (closeOnOverlay) {
                    $popup.on('click', function (e) {
                        if ($(e.target).is($popup)) {
                            self.closePopup($popup);
                        }
                    });
                }

                $(document).on('keydown' + eventNamespace, function (e) {
                    if (e.key === 'Escape' && $popup.hasClass('fb-popup-open')) {
                        self.closePopup($popup);
                    }
                });

                if ($popup.data('disable-auto-trigger')) {
                    return;
                }

                if (trigger === 'time_delay') {
                    var delay = parseInt($popup.data('delay'), 10) || 3;
                    var sessionShown = sessionStorage.getItem(popupShownKey);

                    if (!sessionShown) {
                        setTimeout(function () {
                            if (!isAnyPopupOpen) {
                                self.openPopup($popup);
                                sessionStorage.setItem(popupShownKey, 'true');
                            }
                        }, delay * 1000);
                    }
                } else if (trigger === 'scroll') {
                    var scrollPercentage = parseInt($popup.data('scroll-percentage'), 10) || 50;
                    var scrollTriggered = false;
                    var sessionShown = sessionStorage.getItem(popupShownKey);

                    if (!sessionShown) {
                        $(window).on('scroll' + eventNamespace, function () {
                            if (scrollTriggered || isAnyPopupOpen) return;

                            var scrollTop = $(window).scrollTop();
                            var docHeight = $(document).height() - $(window).height();
                            if (docHeight <= 0) return;
                            var scrollPercent = (scrollTop / docHeight) * 100;

                            if (scrollPercent >= scrollPercentage) {
                                scrollTriggered = true;
                                self.openPopup($popup);
                                sessionStorage.setItem(popupShownKey, 'true');
                                $(window).off('scroll' + eventNamespace);
                            }
                        });
                    }
                } else if (trigger === 'exit_intent') {
                    var exitTriggered = false;
                    var sessionShown = sessionStorage.getItem(popupShownKey);

                    if (!sessionShown) {
                        $(document).on('mouseleave' + eventNamespace, function (e) {
                            if (exitTriggered || isAnyPopupOpen) return;

                            if (e.clientY < 10) {
                                exitTriggered = true;
                                self.openPopup($popup);
                                sessionStorage.setItem(popupShownKey, 'true');
                                $(document).off('mouseleave' + eventNamespace);
                            }
                        });
                    }
                }
            });

            $(document).on('fb:popup:opened', function () {
                isAnyPopupOpen = true;
            });
            $(document).on('fb:popup:closed', function () {
                isAnyPopupOpen = $('.fb-popup-overlay.fb-popup-open').length > 0;
            });
        },

        openPopup: function ($popup) {

            $('.fb-popup-overlay.fb-popup-open').not($popup).each(function () {
                $(this).removeClass('fb-popup-open');
            });

            $popup.addClass('fb-popup-open');
            $('body').addClass('fb-popup-body-open');
            $popup.attr('aria-hidden', 'false');

            $(document).trigger('fb:popup:opened', [$popup]);

            setTimeout(function () {
                var $focusable = $popup.find('input:visible, select:visible, textarea:visible, button:visible').not('.fb-popup-close').first();
                if ($focusable.length) {
                    $focusable.focus();
                }
            }, 150);
        },

        closePopup: function ($popup) {
            $popup.removeClass('fb-popup-open');
            $popup.attr('aria-hidden', 'true');

            if ($('.fb-popup-overlay.fb-popup-open').length === 0) {
                $('body').removeClass('fb-popup-body-open');
            }

            $(document).trigger('fb:popup:closed', [$popup]);
        },

        updateFileList: function ($input, $fileList) {
            var self = this;
            var files = $input[0].files;
            $fileList.empty();

            if (files.length === 0) {
                return;
            }

            for (var i = 0; i < files.length; i++) {
                var file = files[i];
                var fileSize = self.formatFileSize(file.size);

                var $fileItem = $(
                    '<div class="fb-file-item">' +
                    '    <div class="fb-file-item-name">' +
                    '        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">' +
                    '            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>' +
                    '            <polyline points="14 2 14 8 20 8"/>' +
                    '        </svg>' +
                    '        <span>' + self.escapeHtml(file.name) + '</span>' +
                    '        <span class="fb-file-item-size">(' + fileSize + ')</span>' +
                    '    </div>' +
                    '    <button type="button" class="fb-file-item-remove" title="Remove">' +
                    '        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">' +
                    '            <line x1="18" y1="6" x2="6" y2="18"/>' +
                    '            <line x1="6" y1="6" x2="18" y2="18"/>' +
                    '        </svg>' +
                    '    </button>' +
                    '</div>'
                );

                $fileList.append($fileItem);
            }

            $fileList.find('.fb-file-item-remove').on('click', function () {
                $input.val('');
                $fileList.empty();
            });
        },

        formatFileSize: function (bytes) {
            if (bytes === 0) return '0 Bytes';
            var k = 1024;
            var sizes = ['Bytes', 'KB', 'MB', 'GB'];
            var i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        },

        escapeHtml: function (text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        validateStep: function ($step) {
            var self = this;
            var isValid = true;
            var firstInvalidField = null;

            var messages = {
                required: $step.closest('form').data('msg-required') || 'This field is required.',
                email: $step.closest('form').data('msg-email') || 'Please enter a valid email address.',
                phone: $step.closest('form').data('msg-phone') || 'Please enter a valid phone number.',
            };

            var showError = function ($field, message) {
                $field.addClass('is-invalid');
                var $feedback = $field.closest('.fb-form-field').find('.invalid-feedback');
                if ($feedback.length) {
                    $feedback.text(message).show();
                }
            };

            var clearError = function ($field) {
                $field.removeClass('is-invalid');
                var $feedback = $field.closest('.fb-form-field').find('.invalid-feedback');
                if ($feedback.length) {
                    $feedback.text('').hide();
                }
            };

            $step.find('[required]').each(function () {
                var $field = $(this);
                var value = $field.val();
                var fieldType = $field.attr('type');

                if (fieldType === 'checkbox' || fieldType === 'radio') {
                    var fieldName = $field.attr('name');
                    var $group = $step.find('[name="' + fieldName + '"]');
                    var isChecked = $group.filter(':checked').length > 0;

                    if (!isChecked) {
                        isValid = false;
                        $group.addClass('is-invalid');
                        var $feedback = $field.closest('.fb-form-field').find('.invalid-feedback');
                        if ($feedback.length) {
                            $feedback.text(messages.required).show();
                        }
                        if (!firstInvalidField) {
                            firstInvalidField = $field;
                        }
                    } else {
                        $group.removeClass('is-invalid');
                        var $feedback = $field.closest('.fb-form-field').find('.invalid-feedback');
                        if ($feedback.length) {
                            $feedback.text('').hide();
                        }
                    }
                } else {
                    if (!value || (Array.isArray(value) && value.length === 0)) {
                        isValid = false;
                        showError($field, messages.required);
                        if (!firstInvalidField) {
                            firstInvalidField = $field;
                        }
                    } else {
                        clearError($field);
                    }
                }
            });

            $step.find('input[type="email"]').each(function () {
                var $field = $(this);
                var value = $field.val();

                if (value && !self.isValidEmail(value)) {
                    isValid = false;
                    showError($field, messages.email);
                    if (!firstInvalidField) {
                        firstInvalidField = $field;
                    }
                }
            });

            $step.find('input[type="tel"]').each(function () {
                var $field = $(this);
                var value = $field.val();

                if (value && !self.isValidPhone(value)) {
                    isValid = false;
                    showError($field, messages.phone);
                    if (!firstInvalidField) {
                        firstInvalidField = $field;
                    }
                }
            });

            if (firstInvalidField) {
                firstInvalidField.focus();
            }

            return isValid;
        },

        isValidEmail: function (email) {
            var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        },

        isValidPhone: function (phone) {
            var digitsOnly = phone.replace(/\D/g, '');
            return digitsOnly.length >= 6 && /^[\d\s\-\+\(\)\.]+$/.test(phone);
        },

        submitForm: function ($form) {
            var self = this;
            var $wrapper = $form.closest('.fb-form-container');
            if (!$wrapper.length) {
                $wrapper = $form.parent();
            }
            var $submitBtn = $form.find('[type="submit"]:visible');
            if (!$submitBtn.length) {
                $submitBtn = $form.find('[type="submit"]').first();
            }
            var originalBtnHtml = $submitBtn.html();

            var $allPanels = $form.find('.step-content');
            if ($allPanels.length > 0) {
                var allValid = true;
                var firstInvalidPanel = null;

                $allPanels.each(function (index) {
                    if (!self.validateStep($(this))) {
                        allValid = false;
                        if (!firstInvalidPanel) {
                            firstInvalidPanel = index;
                        }
                    }
                });

                if (!allValid) {
                    var $wizard = $form.find('.fb-form-wizard');
                    if ($wizard.length && $wizard.data('fobWizard') && firstInvalidPanel !== null) {
                        $wizard.data('fobWizard').goToStep(firstInvalidPanel);
                    }
                    self.showMessage($wrapper, 'error', 'Please fill in all required fields.');
                    return;
                }
            } else {
                if (!self.validateStep($form)) {
                    self.showMessage($wrapper, 'error', 'Please fill in all required fields.');
                    return;
                }
            }

            $wrapper.addClass('loading');
            $submitBtn.prop('disabled', true).html(
                '<svg class="fb-spinner" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 12a9 9 0 1 1-6.219-8.56"/></svg> Submitting...'
            );

            var formData = new FormData($form[0]);

            $.ajax({
                url: $form.attr('action'),
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function (response) {
                    $wrapper.removeClass('loading');
                    $submitBtn.prop('disabled', false).html(originalBtnHtml);

                    if (response.redirect) {
                        window.location.href = response.redirect;
                        return;
                    }

                    if (response.success) {
                        self.showMessage($wrapper, 'success', response.message);
                        $form[0].reset();

                        var formHash = $wrapper.data('form-hash');
                        if (formHash) {
                            self.clearFormData(formHash);
                        }

                        $wrapper.find('.fb-restore-notification').remove();

                        $form.find('.fb-star-rating').each(function () {
                            var $ratingEl = $(this).find('[data-rateyo]');
                            if ($ratingEl.data('rateYo')) {
                                $ratingEl.rateYo('rating', 0);
                            }
                        });

                        $form.find('.fb-file-list').empty();

                        var $wizard = $form.find('.fb-form-wizard');
                        if ($wizard.length && $wizard.data('fobWizard')) {
                            $wizard.data('fobWizard').reset();
                        }
                    } else {
                        self.showMessage($wrapper, 'error', response.message);
                    }
                },
                error: function (xhr) {
                    $wrapper.removeClass('loading');
                    $submitBtn.prop('disabled', false).html(originalBtnHtml);

                    var message = 'An error occurred. Please try again.';

                    if (xhr.responseJSON) {
                        if (xhr.responseJSON.message) {
                            message = xhr.responseJSON.message;
                        }
                        if (xhr.responseJSON.errors) {
                            var errors = xhr.responseJSON.errors;
                            var errorMessages = [];
                            for (var field in errors) {
                                if (errors.hasOwnProperty(field)) {
                                    errorMessages.push(errors[field][0]);
                                    $form.find('[name="' + field + '"]').addClass('is-invalid');
                                }
                            }
                            if (errorMessages.length > 0) {
                                message = errorMessages.join('<br>');
                            }
                        }
                    }

                    self.showMessage($wrapper, 'error', message);
                }
            });
        },

        showMessage: function ($wrapper, type, message) {
            var $messageContainer = $wrapper.find('.fb-form-message');
            if (!$messageContainer.length) {
                $messageContainer = $('<div class="fb-form-message"></div>');
                $wrapper.find('.fb-form').prepend($messageContainer);
            }

            var alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
            $messageContainer
                .removeClass('alert-success alert-danger')
                .addClass(alertClass)
                .html(message)
                .slideDown(300);

            if (type === 'success') {
                var timeoutSeconds = parseInt($wrapper.data('success-timeout'), 10);
                if (isNaN(timeoutSeconds) || timeoutSeconds < 0) {
                    timeoutSeconds = 5;
                }

                if (timeoutSeconds > 0) {
                    setTimeout(function () {
                        $messageContainer.slideUp(300);
                    }, timeoutSeconds * 1000);
                }
            }

            if ($wrapper.length && $wrapper.offset()) {
                $('html, body').animate({
                    scrollTop: $wrapper.offset().top - 20
                }, 300);
            }
        },

        initUrlTracking: function () {
            var self = this;

            $('.fb-form').each(function () {
                var $form = $(this);
                var $pageUrlInput = $form.find('.fb-page-url');
                var $referrerUrlInput = $form.find('.fb-referrer-url');

                var pageUrl = self.getCurrentPageUrl();
                var referrerUrl = self.getReferrerUrl();

                if ($pageUrlInput.length && pageUrl) {
                    $pageUrlInput.val(pageUrl);
                }

                if ($referrerUrlInput.length && referrerUrl) {
                    $referrerUrlInput.val(referrerUrl);
                }
            });
        },

        getCurrentPageUrl: function () {
            try {
                if (this.isInCrossOriginIframe()) {
                    return document.referrer || window.location.href;
                }
                return window.location.href;
            } catch (e) {
                return window.location.href;
            }
        },

        getReferrerUrl: function () {
            try {
                if (this.isInCrossOriginIframe()) {
                    return '';
                }
                return document.referrer || '';
            } catch (e) {
                return '';
            }
        }
    };

    $(document).ready(function () {
        FbFormPublic.init();
    });

    window.FbFormPublic = FbFormPublic;

})(window.jQuery || window.$);
