(function ($) {
    'use strict';

    const icons = {
        text: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M17 6.1H3"/><path d="M21 12.1H3"/><path d="M15.1 18H3"/></svg>',
        number: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M4 17h6"/><path d="M4 12h6"/><path d="M10 7H4v10"/><path d="M14 7h6v10"/><path d="M14 12h6"/></svg>',
        email: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><rect width="20" height="16" x="2" y="4" rx="2"/><path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7"/></svg>',
        phone: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"/></svg>',
        textarea: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M17 6.1H3"/><path d="M21 12.1H3"/><path d="M21 18H3"/></svg>',
        select: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><rect width="18" height="18" x="3" y="3" rx="2"/><path d="m9 12 2 2 4-4"/></svg>',
        radio: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><circle cx="12" cy="12" r="3"/></svg>',
        checkbox: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><rect width="18" height="18" x="3" y="3" rx="2"/><path d="m9 12 2 2 4-4"/></svg>',
        date: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><rect width="18" height="18" x="3" y="4" rx="2" ry="2"/><line x1="16" x2="16" y1="2" y2="6"/><line x1="8" x2="8" y1="2" y2="6"/><line x1="3" x2="21" y1="10" y2="10"/></svg>',
        time: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/></svg>',
        file: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="17 8 12 3 7 8"/><line x1="12" x2="12" y1="3" y2="15"/></svg>',
        image: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg>',
        hidden: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M9.88 9.88a3 3 0 1 0 4.24 4.24"/><path d="M10.73 5.08A10.43 10.43 0 0 1 12 5c7 0 10 7 10 7a13.16 13.16 0 0 1-1.67 2.68"/><path d="M6.61 6.61A13.526 13.526 0 0 0 2 12s3 7 10 7a9.74 9.74 0 0 0 5.39-1.61"/><line x1="2" x2="22" y1="2" y2="22"/></svg>',
        html: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><polyline points="16 18 22 12 16 6"/><polyline points="8 6 2 12 8 18"/></svg>',
        grip: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="9" cy="5" r="1"/><circle cx="9" cy="12" r="1"/><circle cx="9" cy="19" r="1"/><circle cx="15" cy="5" r="1"/><circle cx="15" cy="12" r="1"/><circle cx="15" cy="19" r="1"/></svg>',
        trash: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 6h18"/><path d="M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6"/><path d="M8 6V4c0-1 1-2 2-2h4c1 0 2 1 2 2v2"/></svg>',
        settings: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12.22 2h-.44a2 2 0 0 0-2 2v.18a2 2 0 0 1-1 1.73l-.43.25a2 2 0 0 1-2 0l-.15-.08a2 2 0 0 0-2.73.73l-.22.38a2 2 0 0 0 .73 2.73l.15.1a2 2 0 0 1 1 1.72v.51a2 2 0 0 1-1 1.74l-.15.09a2 2 0 0 0-.73 2.73l.22.38a2 2 0 0 0 2.73.73l.15-.08a2 2 0 0 1 2 0l.43.25a2 2 0 0 1 1 1.73V20a2 2 0 0 0 2 2h.44a2 2 0 0 0 2-2v-.18a2 2 0 0 1 1-1.73l.43-.25a2 2 0 0 1 2 0l.15.08a2 2 0 0 0 2.73-.73l.22-.39a2 2 0 0 0-.73-2.73l-.15-.08a2 2 0 0 1-1-1.74v-.5a2 2 0 0 1 1-1.74l.15-.09a2 2 0 0 0 .73-2.73l-.22-.38a2 2 0 0 0-2.73-.73l-.15.08a2 2 0 0 1-2 0l-.43-.25a2 2 0 0 1-1-1.73V4a2 2 0 0 0-2-2z"/><circle cx="12" cy="12" r="3"/></svg>',
        close: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="18" x2="6" y1="6" y2="18"/><line x1="6" x2="18" y1="6" y2="18"/></svg>',
        empty: '<svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round"><rect width="18" height="18" x="3" y="3" rx="2"/><path d="M12 8v8"/><path d="M8 12h8"/></svg>',
        chevronUp: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="18 15 12 9 6 15"/></svg>',
        chevronDown: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="6 9 12 15 18 9"/></svg>',
        copy: '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect width="14" height="14" x="8" y="8" rx="2" ry="2"/><path d="M4 16c-1.1 0-2-.9-2-2V4c0-1.1.9-2 2-2h10c1.1 0 2 .9 2 2"/></svg>',
    };

    const fieldTypes = {
        text: { label: 'Text Input', icon: icons.text },
        number: { label: 'Number Input', icon: icons.number },
        email: { label: 'Email', icon: icons.email },
        phone: { label: 'Phone', icon: icons.phone },
        textarea: { label: 'Multi-line Input', icon: icons.textarea },
        autocomplete: { label: 'Autocomplete', icon: icons.text },
        checkbox: { label: 'Checkbox', icon: icons.checkbox },
        radio: { label: 'Radio Button', icon: icons.radio },
        select: { label: 'Dropdown', icon: icons.select },
        date: { label: 'Date', icon: icons.date },
        time: { label: 'Time', icon: icons.time },
        file: { label: 'File Upload', icon: icons.file },
        image: { label: 'Image Upload', icon: icons.image },
        hidden: { label: 'Hidden', icon: icons.hidden },
        html: { label: 'HTML Content', icon: icons.html },
    };

    var FbFormBuilder = {
        steps: [],
        activeStep: 0,
        selectedField: null,
        keyboardGrabbedField: null,
        originalFieldPosition: null,
        deleteConfirmPending: null,

        history: [],
        historyIndex: -1,
        maxHistory: 20,
        isUndoRedo: false,
        updateFieldDebounce: null,

        init: function () {
            this.renderComponentPalette();
            this.initFormBuilder();
            this.initActions();
            this.bindEvents();

            this.loadDynamicOptions($('#actions-container'));
        },

        renderComponentPalette: function () {
            var $palette = $('#fb-component-palette');
            if (!$palette.length) return;

            var html = '';
            Object.keys(fieldTypes).forEach(function (type) {
                var config = fieldTypes[type];
                html += '<div class="fb-component-card" data-field-type="' + type + '">' +
                    '<div class="fb-component-icon">' + config.icon + '</div>' +
                    '<div class="fb-component-label">' + config.label + '</div>' +
                    '</div>';
            });

            $palette.html(html);
        },

        initFormBuilder: function () {
            var self = this;
            var $container = $('#bb-form-builder');
            if (!$container.length) return;

            var existingContent = (window.fbFormBuilderConfig && window.fbFormBuilderConfig.formContent) ? window.fbFormBuilderConfig.formContent : ($container.data('content') || {});

            // Load step titles from properties
            var stepTitles = {};
            $('input[name^="properties[step_titles]"]').each(function() {
                var match = $(this).attr('name').match(/properties\[step_titles\]\[([^\]]+)\]/);
                if (match) {
                    stepTitles[match[1]] = $(this).val();
                }
            });

            // Load step conditions from properties
            var stepConditions = {};
            $('input[name^="properties[step_conditions]"]').each(function() {
                var match = $(this).attr('name').match(/properties\[step_conditions\]\[([^\]]+)\]\[([^\]]+)\]/);
                if (match) {
                    var stepId = match[1];
                    var prop = match[2];
                    if (!stepConditions[stepId]) {
                        stepConditions[stepId] = {};
                    }
                    stepConditions[stepId][prop] = $(this).val();
                }
            });

            if (typeof existingContent === 'object' && !Array.isArray(existingContent)) {
                var stepKeys = Object.keys(existingContent);
                if (stepKeys.length > 0) {
                    this.steps = stepKeys.map(function (key) {
                        // Use saved title from properties, or generate from key
                        var title = stepTitles[key] || key.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
                        var step = {
                            id: key,
                            title: title,
                            fields: existingContent[key] || []
                        };
                        // Load condition if exists
                        if (stepConditions[key] && stepConditions[key].field) {
                            step.condition = stepConditions[key];
                        }
                        return step;
                    });
                } else {
                    this.steps = [{ id: 'step-1', title: 'Step 1', fields: [] }];
                }
            } else if (Array.isArray(existingContent) && existingContent.length > 0) {
                var title = stepTitles['step-1'] || 'Step 1';
                this.steps = [{ id: 'step-1', title: title, fields: existingContent }];
            } else {
                this.steps = [{ id: 'step-1', title: 'Step 1', fields: [] }];
            }

            this.renderSteps();

            this.saveToHistory('initial');
        },

        renderSteps: function () {
            var self = this;

            var $nav = $('#fb-steps-nav');
            var tabsHtml = '';
            this.steps.forEach(function (step, index) {
                var isActive = index === self.activeStep;
                var hasCondition = step.condition && step.condition.field;
                tabsHtml += '<div class="fb-step-tab' + (isActive ? ' active' : '') + (hasCondition ? ' has-condition' : '') + '" data-step="' + index + '">' +
                    '<span class="step-title" data-step="' + index + '" title="Double-click to edit">' + self.escapeHtml(step.title) + '</span>' +
                    (self.steps.length > 1 ? '<span class="step-remove" data-step="' + index + '">' + icons.close + '</span>' : '') +
                    '</div>';
            });
            $nav.html(tabsHtml);

            // Update hidden inputs for step titles
            this.saveStepTitles();

            var $container = $('#fb-steps-container');
            var panelsHtml = '';
            this.steps.forEach(function (step, index) {
                var isActive = index === self.activeStep;
                var hasCondition = step.condition && step.condition.field;
                var canHaveCondition = index > 0; // Only steps after the first can have conditions
                var conditionArea = '';

                // Rename step button (always show)
                var renameBtn = '<button type="button" class="fb-step-action-btn fb-rename-step-btn" data-step="' + index + '">' +
                    '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17 3a2.85 2.83 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5Z"/></svg>' +
                    '<span>Rename step</span>' +
                    '</button>';

                if (hasCondition) {
                    var conditionField = step.condition.field;
                    var conditionOperator = step.condition.operator;
                    var operatorText = conditionOperator === 'checked' ? 'is checked' :
                        conditionOperator === 'not_checked' ? 'is not checked' :
                        conditionOperator === 'equals' ? '= "' + self.escapeHtml(step.condition.value) + '"' :
                        '≠ "' + self.escapeHtml(step.condition.value) + '"';
                    conditionArea = '<div class="fb-step-actions-bar">' + renameBtn +
                        '<div class="fb-step-condition-badge" data-step="' + index + '">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M16 3h5v5"/><path d="M4 20 21 3"/><path d="M21 16v5h-5"/><path d="M15 15l6 6"/><path d="M4 4l5 5"/></svg>' +
                        '<span>Show if <strong>' + self.escapeHtml(conditionField) + '</strong> ' + operatorText + '</span>' +
                        '<button type="button" class="fb-condition-badge-edit" data-step="' + index + '" title="Edit condition">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17 3a2.85 2.83 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5Z"/></svg>' +
                        '</button>' +
                        '</div></div>';
                } else if (canHaveCondition) {
                    conditionArea = '<div class="fb-step-actions-bar">' + renameBtn +
                        '<button type="button" class="fb-step-action-btn fb-add-condition-btn" data-step="' + index + '">' +
                        '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M16 3h5v5"/><path d="M4 20 21 3"/><path d="M21 16v5h-5"/><path d="M15 15l6 6"/><path d="M4 4l5 5"/></svg>' +
                        '<span>Add condition</span>' +
                        '</button></div>';
                } else {
                    // First step - only rename button
                    conditionArea = '<div class="fb-step-actions-bar">' + renameBtn + '</div>';
                }

                panelsHtml += '<div class="fb-step-panel' + (isActive ? ' active' : '') + '" data-step="' + index + '">' +
                    conditionArea +
                    '<div class="fb-fields-container' + (step.fields.length > 0 ? ' has-fields' : '') + '" data-step="' + index + '">' +
                    self.renderFields(step.fields, index) +
                    '</div>' +
                    '</div>';
            });
            $container.html(panelsHtml);

            this.initSortable();
            this.saveFormData();
        },

        renderFields: function (fields, stepIndex) {
            var self = this;

            if (!fields || fields.length === 0) {
                return '<div class="fb-empty-state">' +
                    '<div class="empty-icon">' + icons.empty + '</div>' +
                    '<div class="empty-title">No fields yet</div>' +
                    '<div class="empty-hint">Click a component from the left panel to add fields</div>' +
                    '</div>';
            }

            var html = '';
            fields.forEach(function (field, fieldIndex) {
                html += self.renderFieldItem(field, stepIndex, fieldIndex);
            });

            return html;
        },

        renderFieldItem: function (field, stepIndex, fieldIndex) {
            var isSelected = this.selectedField && this.selectedField.step === stepIndex && this.selectedField.field === fieldIndex;
            var hasLabel = field.label && field.label.trim() !== '';
            var labelText = hasLabel ? field.label : 'Untitled';
            var requiredMark = field.required ? '<span class="required-mark">*</span>' : '';
            var hiddenNote = field.hide_label ? '<span class="text-muted small ms-2">(label hidden)</span>' : '';

            var isGrabbed = this.keyboardGrabbedField &&
                this.keyboardGrabbedField.step === stepIndex &&
                this.keyboardGrabbedField.field === fieldIndex;

            var totalFields = this.steps[stepIndex].fields.length;
            var isFirst = fieldIndex === 0;
            var isLast = fieldIndex === totalFields - 1;

            var html = '<div class="fb-field-item' +
                (isSelected ? ' is-selected' : '') +
                (isGrabbed ? ' is-keyboard-grabbed' : '') +
                '" data-step="' + stepIndex + '"' +
                ' data-field="' + fieldIndex + '"' +
                ' tabindex="0"' +
                ' role="listitem"' +
                ' aria-grabbed="' + (isGrabbed ? 'true' : 'false') + '"' +
                ' aria-label="' + this.escapeHtml(labelText) + ' field, position ' + (fieldIndex + 1) + ' of ' + totalFields + '">' +
                '<div class="fb-field-inner">' +
                '<div class="fb-field-drag">' + icons.grip + '</div>' +
                '<div class="fb-field-content">' +
                '<div class="fb-field-label">' +
                '<span class="fb-editable fb-editable-label" ' +
                'data-prop="label" ' +
                'data-step="' + stepIndex + '" ' +
                'data-field="' + fieldIndex + '" ' +
                'data-placeholder="Enter label..." ' +
                'contenteditable="false">' +
                this.escapeHtml(labelText) +
                '</span>' +
                requiredMark +
                hiddenNote +
                '</div>' +
                '<div class="fb-field-preview">' +
                this.renderFieldPreviewEditable(field, stepIndex, fieldIndex) +
                '</div>' +
                '</div>' +
                '<div class="fb-field-actions">' +
                '<button type="button" class="btn-field-action btn-move-up" ' +
                'data-step="' + stepIndex + '" data-field="' + fieldIndex + '" ' +
                'title="Move up" aria-label="Move field up"' +
                (isFirst ? ' disabled' : '') + '>' + icons.chevronUp + '</button>' +
                '<button type="button" class="btn-field-action btn-move-down" ' +
                'data-step="' + stepIndex + '" data-field="' + fieldIndex + '" ' +
                'title="Move down" aria-label="Move field down"' +
                (isLast ? ' disabled' : '') + '>' + icons.chevronDown + '</button>' +
                '<button type="button" class="btn-field-action btn-duplicate" ' +
                'data-step="' + stepIndex + '" data-field="' + fieldIndex + '" ' +
                'title="Duplicate" aria-label="Duplicate field">' + icons.copy + '</button>' +
                '<button type="button" class="btn-field-action btn-settings" data-step="' + stepIndex + '" data-field="' + fieldIndex + '">' + icons.settings + '</button>' +
                '<button type="button" class="btn-field-action btn-delete" data-step="' + stepIndex + '" data-field="' + fieldIndex + '">' + icons.trash + '</button>' +
                '</div>' +
                '</div>' +
                '</div>';

            return html;
        },

        renderFieldPreview: function (field) {
            var type = field.type;
            var placeholder = field.placeholder || '';

            switch (type) {
                case 'text':
                case 'email':
                case 'phone':
                case 'number':
                case 'date':
                case 'time':
                case 'autocomplete':
                    return '<div class="preview-input">' + this.escapeHtml(placeholder || 'Enter ' + type + '...') + '</div>';

                case 'textarea':
                    return '<div class="preview-textarea">' + this.escapeHtml(placeholder || 'Enter text...') + '</div>';

                case 'select':
                    var selectOptions = field.options || ['Option 1', 'Option 2'];
                    return '<select class="preview-select" disabled>' +
                        '<option>' + this.escapeHtml(placeholder || 'Select an option') + '</option>' +
                        '</select>';

                case 'radio':
                    var radioOptions = field.options || ['Option 1', 'Option 2', 'Option 3'];
                    var radioHtml = '<div class="preview-options">';
                    radioOptions.slice(0, 3).forEach(function (opt) {
                        var label = typeof opt === 'object' ? (opt.label || opt.value) : opt;
                        radioHtml += '<label class="preview-option"><input type="radio" disabled> ' + this.escapeHtml(label) + '</label>';
                    }.bind(this));
                    radioHtml += '</div>';
                    return radioHtml;

                case 'checkbox':
                    var checkOptions = field.options || ['Option 1', 'Option 2', 'Option 3'];
                    var checkHtml = '<div class="preview-options">';
                    checkOptions.slice(0, 3).forEach(function (opt) {
                        var label = typeof opt === 'object' ? (opt.label || opt.value) : opt;
                        checkHtml += '<label class="preview-option"><input type="checkbox" disabled> ' + this.escapeHtml(label) + '</label>';
                    }.bind(this));
                    checkHtml += '</div>';
                    return checkHtml;

                case 'file':
                case 'image':
                    return '<div class="preview-input">Click to upload ' + (type === 'image' ? 'image' : 'file') + '...</div>';

                case 'hidden':
                    return '<div class="preview-input" style="opacity:0.5;font-style:italic;">Hidden field</div>';

                case 'html':
                    var content = field.content || '<p>HTML content here...</p>';
                    return '<div class="preview-html" style="padding:8px;background:#f8f9fa;border-radius:4px;font-size:12px;color:#6b7280;border:1px dashed #e5e7eb;">' + content + '</div>';

                default:
                    return '<div class="preview-input">' + this.escapeHtml(placeholder || 'Enter value...') + '</div>';
            }
        },

        renderFieldPreviewEditable: function (field, stepIndex, fieldIndex) {
            var type = field.type;
            var placeholder = field.placeholder || '';
            var self = this;

            var editablePlaceholder = function (defaultText) {
                return '<span class="fb-editable fb-editable-placeholder" ' +
                    'data-prop="placeholder" ' +
                    'data-step="' + stepIndex + '" ' +
                    'data-field="' + fieldIndex + '" ' +
                    'data-placeholder="Enter placeholder..." ' +
                    'contenteditable="false">' +
                    self.escapeHtml(placeholder || defaultText) +
                    '</span>';
            };

            var editableOptions = function (options, inputType) {
                var optionsHtml = '<div class="preview-options">';
                (options || ['Option 1', 'Option 2', 'Option 3']).slice(0, 3).forEach(function (opt, idx) {
                    var label = typeof opt === 'object' ? (opt.label || opt.value) : opt;
                    optionsHtml += '<label class="preview-option">' +
                        '<input type="' + inputType + '" disabled> ' +
                        '<span class="fb-editable fb-editable-option" ' +
                        'data-prop="option" ' +
                        'data-option-index="' + idx + '" ' +
                        'data-step="' + stepIndex + '" ' +
                        'data-field="' + fieldIndex + '" ' +
                        'data-placeholder="Option" ' +
                        'contenteditable="false">' +
                        self.escapeHtml(label) +
                        '</span>' +
                        '</label>';
                });
                optionsHtml += '</div>';
                return optionsHtml;
            };

            switch (type) {
                case 'text':
                case 'email':
                case 'phone':
                case 'number':
                case 'date':
                case 'time':
                case 'autocomplete':
                    return '<div class="preview-input">' + editablePlaceholder('Enter ' + type + '...') + '</div>';

                case 'textarea':
                    return '<div class="preview-textarea">' + editablePlaceholder('Enter text...') + '</div>';

                case 'select':
                    return '<select class="preview-select" disabled>' +
                        '<option>' + this.escapeHtml(placeholder || 'Select an option') + '</option>' +
                        '</select>';

                case 'radio':
                    return editableOptions(field.options, 'radio');

                case 'checkbox':
                    return editableOptions(field.options, 'checkbox');

                case 'file':
                case 'image':
                    return '<div class="preview-input">Click to upload ' + (type === 'image' ? 'image' : 'file') + '...</div>';

                case 'hidden':
                    return '<div class="preview-input" style="opacity:0.5;font-style:italic;">Hidden field</div>';

                case 'html':
                    var content = field.content || '<p>HTML content here...</p>';
                    return '<div class="preview-html" style="padding:8px;background:#f8f9fa;border-radius:4px;font-size:12px;color:#6b7280;border:1px dashed #e5e7eb;">' + content + '</div>';

                default:
                    return '<div class="preview-input">' + editablePlaceholder('Enter value...') + '</div>';
            }
        },

        initSortable: function () {
            var self = this;

            $('.fb-fields-container').each(function () {
                var $container = $(this);

                if ($container.data('ui-sortable')) {
                    $container.sortable('destroy');
                }

                if ($container.data('ui-droppable')) {
                    $container.droppable('destroy');
                }

                $container.sortable({
                    handle: '.fb-field-drag',
                    items: '.fb-field-item',
                    placeholder: 'fb-field-placeholder',
                    tolerance: 'pointer',
                    cursor: 'grabbing',
                    opacity: 1,
                    revert: false,
                    scrollSensitivity: 60,
                    scrollSpeed: 20,

                    start: function (event, ui) {
                        ui.placeholder.height(ui.item.outerHeight());
                        $container.addClass('ui-sortable-sorting');
                        ui.item.data('startIndex', ui.item.index());
                        $('body').addClass('is-dragging-field');
                    },

                    over: function (event, ui) {
                        $(this).addClass('is-drag-over');
                    },

                    out: function (event, ui) {
                        $(this).removeClass('is-drag-over');
                    },

                    stop: function (event, ui) {
                        $container.removeClass('ui-sortable-sorting is-drag-over');
                        $('body').removeClass('is-dragging-field');

                        if (ui.item.hasClass('fb-field-item')) {
                            ui.item.addClass('just-dropped');
                            setTimeout(function () {
                                ui.item.removeClass('just-dropped');
                            }, 150);
                        }
                    },

                    update: function (event, ui) {

                        if (ui.sender) return;

                        self.saveToHistory('reorder_field');

                        var stepIndex = $container.data('step');
                        var newOrder = [];

                        $container.find('.fb-field-item').each(function () {
                            var oldIndex = $(this).data('field');
                            newOrder.push(self.steps[stepIndex].fields[oldIndex]);
                        });

                        self.steps[stepIndex].fields = newOrder;
                        self.renderSteps();
                    }
                });

                $container.droppable({
                    accept: '.fb-component-card',
                    tolerance: 'pointer',
                    over: function (event, ui) {
                        $(this).addClass('is-drag-over is-drop-target');
                    },
                    out: function (event, ui) {
                        $(this).removeClass('is-drag-over');
                    },
                    drop: function (event, ui) {
                        var $cont = $(this);
                        var stepIndex = $cont.data('step');
                        var fieldType = ui.draggable.data('field-type');

                        var dropY = event.pageY;
                        var insertIndex = self.getInsertIndexFromPosition($cont, dropY);

                        $cont.removeClass('is-drag-over is-drop-target');

                        self.addFieldAtPosition(stepIndex, fieldType, insertIndex);
                    }
                });
            });

            this.initPaletteDraggable();
        },

        initPaletteDraggable: function () {
            var self = this;

            $('.fb-component-card').each(function () {
                if ($(this).data('ui-draggable')) {
                    $(this).draggable('destroy');
                }
            });

            $('.fb-component-card').draggable({
                helper: function () {
                    var $this = $(this);
                    var fieldType = $this.data('field-type');
                    var config = fieldTypes[fieldType];

                    if (!config) {
                        console.error('Invalid field type:', fieldType);
                        return $('<div class="fb-drag-helper">Unknown field</div>');
                    }

                    return $('<div class="fb-drag-helper">' +
                        '<div class="fb-drag-helper-icon">' + config.icon + '</div>' +
                        '<div class="fb-drag-helper-label">' + config.label + '</div>' +
                        '</div>');
                },
                appendTo: 'body',
                zIndex: 10000,
                revert: 'invalid',
                revertDuration: 150,
                cursor: 'grabbing',
                cursorAt: { top: 25, left: 50 },
                scroll: false,
                start: function (event, ui) {
                    var $this = $(this);
                    $this.addClass('is-dragging-source');
                    $('body').addClass('is-dragging-from-palette');

                    $('.fb-fields-container').addClass('is-drop-target');
                },
                stop: function (event, ui) {
                    var $this = $(this);
                    $this.removeClass('is-dragging-source');
                    $('body').removeClass('is-dragging-from-palette');

                    $('.fb-fields-container').removeClass('is-drop-target is-drag-over');
                }
            });
        },

        addField: function (stepIndex, fieldType) {

            this.saveToHistory('add_field');

            var config = fieldTypes[fieldType];
            var field = {
                type: fieldType,
                label: config.label,
                name: fieldType + '_' + Date.now(),
                required: false,
                hide_label: false,
                placeholder: '',
                width: '12'
            };

            if (['select', 'radio', 'checkbox', 'autocomplete'].includes(fieldType)) {
                field.options = ['Option 1', 'Option 2', 'Option 3'];
            }

            if (fieldType === 'phone') {
                field.enable_country_code = true;
            }

            if (fieldType === 'file') {
                field.accept = '.pdf,.doc,.docx,.xls,.xlsx';
            }

            if (fieldType === 'image') {
                field.accept = '.jpg,.jpeg,.png,.gif,.webp';
            }

            if (fieldType === 'html') {
                field.label = 'HTML Content';
                field.content = '<p>Enter your HTML content here...</p>';
                delete field.name;
                delete field.required;
                delete field.placeholder;
            }

            this.steps[stepIndex].fields.push(field);
            this.renderSteps();

            var fieldIndex = this.steps[stepIndex].fields.length - 1;
            this.selectField(stepIndex, fieldIndex);
        },

        addFieldAtPosition: function (stepIndex, fieldType, position) {

            this.saveToHistory('add_field');

            var config = fieldTypes[fieldType];
            var field = {
                type: fieldType,
                label: config.label,
                name: fieldType + '_' + Date.now(),
                required: false,
                hide_label: false,
                placeholder: '',
                width: '12'
            };

            if (['select', 'radio', 'checkbox', 'autocomplete'].includes(fieldType)) {
                field.options = ['Option 1', 'Option 2', 'Option 3'];
            }

            if (fieldType === 'phone') {
                field.enable_country_code = true;
            }

            if (fieldType === 'file') {
                field.accept = '.pdf,.doc,.docx,.xls,.xlsx';
            }

            if (fieldType === 'image') {
                field.accept = '.jpg,.jpeg,.png,.gif,.webp';
            }

            if (fieldType === 'html') {
                field.label = 'HTML Content';
                field.content = '<p>Enter your HTML content here...</p>';
                delete field.name;
                delete field.required;
                delete field.placeholder;
            }

            var fields = this.steps[stepIndex].fields;
            var actualIndex;
            if (position >= 0 && position <= fields.length) {
                fields.splice(position, 0, field);
                actualIndex = position;
            } else {
                fields.push(field);
                actualIndex = fields.length - 1;
            }

            this.renderSteps();

            this.selectField(stepIndex, actualIndex);
        },

        getInsertIndexFromPosition: function ($container, dropY) {
            var $items = $container.find('.fb-field-item');
            var insertIndex = $items.length; // Default to end

            $items.each(function (index) {
                var $item = $(this);
                var itemTop = $item.offset().top;
                var itemHeight = $item.outerHeight();
                var itemMiddle = itemTop + (itemHeight / 2);

                if (dropY < itemMiddle) {
                    insertIndex = index;
                    return false; // Break loop
                }
            });

            return insertIndex;
        },

        removeField: function (stepIndex, fieldIndex) {

            this.saveToHistory('delete_field');

            this.steps[stepIndex].fields.splice(fieldIndex, 1);
            this.closeSettings();
            this.renderSteps();
        },

        selectField: function (stepIndex, fieldIndex) {
            this.selectedField = { step: stepIndex, field: fieldIndex };
            this.renderSteps();
            this.showSettings(stepIndex, fieldIndex);
        },

        showSettings: function (stepIndex, fieldIndex) {
            var field = this.steps[stepIndex].fields[fieldIndex];
            var self = this;
            var html = '';

            if (field.type === 'html') {
                html = '<div class="fb-setting-group">' +
                    '<label class="setting-label">Label (for admin)</label>' +
                    '<input type="text" class="form-control field-setting" data-prop="label" value="' + this.escapeHtml(field.label || '') + '">' +
                    '</div>' +
                    '<div class="fb-setting-group">' +
                    '<label class="setting-label">HTML Content</label>' +
                    '<textarea class="form-control field-setting" data-prop="content" rows="8" style="font-family:monospace;font-size:12px;">' + this.escapeHtml(field.content || '') + '</textarea>' +
                    '<small class="text-muted">Enter HTML code. Be careful with scripts.</small>' +
                    '</div>' +
                    '<div class="fb-setting-group">' +
                    '<label class="setting-label">Width</label>' +
                    '<div class="fb-width-selector">' +
                    '<button type="button" class="width-option' + (field.width === '3' ? ' active' : '') + '" data-width="3">1/4</button>' +
                    '<button type="button" class="width-option' + (field.width === '4' ? ' active' : '') + '" data-width="4">1/3</button>' +
                    '<button type="button" class="width-option' + (field.width === '6' ? ' active' : '') + '" data-width="6">1/2</button>' +
                    '<button type="button" class="width-option' + ((field.width === '12' || !field.width) ? ' active' : '') + '" data-width="12">Full</button>' +
                    '</div>' +
                    '</div>';

                $('#fb-field-settings-body').html(html);
                $('#fb-field-settings').show();
                return;
            }

            html = '<div class="fb-setting-group">' +
                '<label class="setting-label">Label</label>' +
                '<input type="text" class="form-control field-setting" data-prop="label" value="' + this.escapeHtml(field.label || '') + '">' +
                '</div>' +
                '<div class="fb-setting-group">' +
                '<label class="setting-label">Field Name</label>' +
                '<input type="text" class="form-control field-setting" data-prop="name" value="' + this.escapeHtml(field.name || '') + '">' +
                '</div>' +
                '<div class="fb-setting-group">' +
                '<label class="setting-label">Placeholder</label>' +
                '<input type="text" class="form-control field-setting" data-prop="placeholder" value="' + this.escapeHtml(field.placeholder || '') + '">' +
                '</div>';

            if (['select', 'radio'].includes(field.type)) {
                var options = field.options || [];
                var defaultVal = field.default_value || '';
                html += '<div class="fb-setting-group">' +
                    '<label class="setting-label">Default Value</label>' +
                    '<select class="form-select field-setting" data-prop="default_value">' +
                    '<option value="">-- None --</option>';
                options.forEach(function (opt) {
                    var optVal = typeof opt === 'object' ? (opt.value || opt.label || '') : opt;
                    var optLabel = typeof opt === 'object' ? (opt.label || opt.value || '') : opt;
                    var selected = defaultVal === optVal ? ' selected' : '';
                    html += '<option value="' + self.escapeHtml(optVal) + '"' + selected + '>' + self.escapeHtml(optLabel) + '</option>';
                });
                html += '</select></div>';
            } else if (['checkbox', 'checkbox-group'].includes(field.type)) {
                var options = field.options || [];
                var defaultVals = field.default_value || [];
                if (!Array.isArray(defaultVals)) defaultVals = defaultVals ? [defaultVals] : [];
                html += '<div class="fb-setting-group">' +
                    '<label class="setting-label">Default Selected</label>' +
                    '<div class="fb-default-checkboxes">';
                options.forEach(function (opt, idx) {
                    var optVal = typeof opt === 'object' ? (opt.value || opt.label || '') : opt;
                    var optLabel = typeof opt === 'object' ? (opt.label || opt.value || '') : opt;
                    var checked = defaultVals.includes(optVal) ? ' checked' : '';
                    html += '<label class="form-check d-block">' +
                        '<input type="checkbox" class="form-check-input field-setting-checkbox" data-prop="default_value" value="' + self.escapeHtml(optVal) + '"' + checked + '>' +
                        '<span class="form-check-label">' + self.escapeHtml(optLabel) + '</span>' +
                        '</label>';
                });
                html += '</div></div>';
            } else if (!['file', 'image'].includes(field.type)) {
                var inputType = 'text';
                if (field.type === 'number') inputType = 'number';
                if (field.type === 'date') inputType = 'date';
                if (field.type === 'time') inputType = 'time';
                if (field.type === 'datetime') inputType = 'datetime-local';

                html += '<div class="fb-setting-group">' +
                    '<label class="setting-label">Default Value</label>' +
                    '<input type="' + inputType + '" class="form-control field-setting" data-prop="default_value" value="' + this.escapeHtml(field.default_value || '') + '">' +
                    '</div>';
            }

            html += '<div class="fb-setting-group">' +
                '<label class="setting-label">Width</label>' +
                '<div class="fb-width-selector">' +
                '<button type="button" class="width-option' + (field.width === '3' ? ' active' : '') + '" data-width="3">1/4</button>' +
                '<button type="button" class="width-option' + (field.width === '4' ? ' active' : '') + '" data-width="4">1/3</button>' +
                '<button type="button" class="width-option' + (field.width === '6' ? ' active' : '') + '" data-width="6">1/2</button>' +
                '<button type="button" class="width-option' + ((field.width === '12' || !field.width) ? ' active' : '') + '" data-width="12">Full</button>' +
                '</div>' +
                '</div>' +
                '<div class="fb-setting-group">' +
                '<label class="form-check">' +
                '<input type="checkbox" class="form-check-input field-setting" data-prop="required"' + (field.required ? ' checked' : '') + '>' +
                '<span class="form-check-label">Required field</span>' +
                '</label>' +
                '</div>' +
                '<div class="fb-setting-group">' +
                '<label class="form-check">' +
                '<input type="checkbox" class="form-check-input field-setting" data-prop="hide_label"' + (field.hide_label ? ' checked' : '') + '>' +
                '<span class="form-check-label">Hide label</span>' +
                '</label>' +
                '</div>';

            if (['select', 'radio', 'checkbox', 'autocomplete'].includes(field.type)) {
                var optionsText = (field.options || []).map(function (opt) {
                    return typeof opt === 'object' ? (opt.label || opt.value) : opt;
                }).join('\n');

                html += '<div class="fb-setting-group">' +
                    '<label class="setting-label">Options (one per line)</label>' +
                    '<textarea class="form-control field-setting" data-prop="options" rows="4">' + this.escapeHtml(optionsText) + '</textarea>' +
                    '<small class="text-muted">After changing options, update Default Value above if needed.</small>' +
                    '</div>';
            }

            if (field.type === 'phone') {
                var enableCountryCode = field.enable_country_code !== false;
                html += '<div class="fb-setting-group">' +
                    '<label class="form-check">' +
                    '<input type="checkbox" class="form-check-input field-setting" data-prop="enable_country_code"' + (enableCountryCode ? ' checked' : '') + '>' +
                    '<span class="form-check-label">Enable country code selector</span>' +
                    '</label>' +
                    '</div>';
            }

            if (field.type === 'file' || field.type === 'image') {
                var defaultAccept = field.type === 'image' ? '.jpg,.jpeg,.png,.gif,.webp' : '.pdf,.doc,.docx,.xls,.xlsx';
                html += '<div class="fb-setting-group">' +
                    '<label class="setting-label">Allowed file types</label>' +
                    '<input type="text" class="form-control field-setting" data-prop="accept" value="' + this.escapeHtml(field.accept || defaultAccept) + '">' +
                    '<small class="text-muted">Comma-separated extensions, e.g.: .pdf,.doc,.docx</small>' +
                    '</div>';
            }

            $('#fb-field-settings-body').html(html);
            $('#fb-field-settings').show();
        },

        closeSettings: function () {
            this.selectedField = null;
            $('#fb-field-settings').hide();
            this.renderSteps();
        },

        updateField: function (prop, value) {
            if (!this.selectedField) return;

            var field = this.steps[this.selectedField.step].fields[this.selectedField.field];

            var self = this;
            if (!this.updateFieldDebounce) {
                this.saveToHistory('edit_field');
            }
            clearTimeout(this.updateFieldDebounce);
            this.updateFieldDebounce = setTimeout(function () {
                self.updateFieldDebounce = null;
            }, 1000); // Group edits within 1 second

            if (prop === 'options') {
                field.options = value.split('\n').filter(function (v) { return v.trim() !== ''; });
            } else if (prop === 'required') {
                field.required = value;
            } else if (prop === 'hide_label') {
                field.hide_label = value;
            } else if (prop === 'content') {
                field.content = value;
            } else if (prop === 'enable_country_code') {
                field.enable_country_code = value;
            } else {
                field[prop] = value;
            }

            this.saveFormData();

            if (!$('.fb-editable.is-editing').length) {
                this.renderSteps();
            } else {

                var $inlineLabel = $('.fb-editable-label[data-step="' + this.selectedField.step + '"][data-field="' + this.selectedField.field + '"]');
                if (prop === 'label' && !$inlineLabel.hasClass('is-editing')) {
                    $inlineLabel.text(value || 'Untitled');
                }
            }
        },

        saveInlineEdit: function ($editable) {
            if (!$editable.hasClass('is-editing')) return;

            var stepIndex = parseInt($editable.data('step'));
            var fieldIndex = parseInt($editable.data('field'));
            var prop = $editable.data('prop');
            var value = $editable.text().trim();

            if (prop === 'option') {
                var optionIndex = parseInt($editable.data('option-index'));
                var field = this.steps[stepIndex].fields[fieldIndex];
                var options = field.options || [];

                if (options[optionIndex] !== undefined) {
                    if (typeof options[optionIndex] === 'object') {
                        options[optionIndex].label = value;
                    } else {
                        options[optionIndex] = value;
                    }
                    field.options = options;
                }
            } else {

                this.steps[stepIndex].fields[fieldIndex][prop] = value;
            }

            $editable.attr('contenteditable', 'false')
                .removeClass('is-editing');

            if (this.selectedField &&
                this.selectedField.step === stepIndex &&
                this.selectedField.field === fieldIndex) {
                var $settingInput = $('.field-setting[data-prop="' + prop + '"]');
                if ($settingInput.length) {
                    $settingInput.val(value);
                }
            }

            this.saveFormData();
        },

        cancelInlineEdit: function ($editable) {
            var stepIndex = parseInt($editable.data('step'));
            var fieldIndex = parseInt($editable.data('field'));
            var prop = $editable.data('prop');
            var field = this.steps[stepIndex].fields[fieldIndex];

            var originalValue = '';
            if (prop === 'option') {
                var optionIndex = parseInt($editable.data('option-index'));
                var opt = (field.options || [])[optionIndex];
                originalValue = typeof opt === 'object' ? (opt.label || opt.value) : (opt || '');
            } else {
                originalValue = field[prop] || '';
            }

            $editable.text(originalValue)
                .attr('contenteditable', 'false')
                .removeClass('is-editing');
        },

        exitInlineEdit: function () {
            var $activeEdit = $('.fb-editable.is-editing');
            if ($activeEdit.length) {
                this.saveInlineEdit($activeEdit);
            }
        },

        toggleKeyboardGrab: function (stepIndex, fieldIndex) {
            if (this.keyboardGrabbedField &&
                this.keyboardGrabbedField.step === stepIndex &&
                this.keyboardGrabbedField.field === fieldIndex) {

                this.keyboardGrabbedField = null;
                this.originalFieldPosition = null;
                this.renderSteps();
                this.announce(this.steps[stepIndex].fields[fieldIndex].label + ' field dropped');

                var self = this;
                setTimeout(function () {
                    $('.fb-field-item[data-step="' + stepIndex + '"][data-field="' + fieldIndex + '"]').focus();
                }, 50);
            } else {

                this.saveToHistory('reorder_field');
                this.keyboardGrabbedField = { step: stepIndex, field: fieldIndex };
                this.originalFieldPosition = fieldIndex;
                this.renderSteps();

                var fieldLabel = this.steps[stepIndex].fields[fieldIndex].label || 'Untitled';
                this.announce(fieldLabel + ' field grabbed. Use arrow keys to move, Space to drop.');

                var self = this;
                setTimeout(function () {
                    $('.fb-field-item[data-step="' + stepIndex + '"][data-field="' + fieldIndex + '"]').focus();
                }, 50);
            }
        },

        moveFieldKeyboard: function (stepIndex, fieldIndex, direction) {
            var fields = this.steps[stepIndex].fields;
            var newIndex = fieldIndex + direction;

            if (newIndex < 0 || newIndex >= fields.length) {
                return; // Can't move further
            }

            var temp = fields[fieldIndex];
            fields[fieldIndex] = fields[newIndex];
            fields[newIndex] = temp;

            this.keyboardGrabbedField.field = newIndex;

            var fieldLabel = temp.label || 'Untitled';
            this.announce(fieldLabel + ' moved to position ' + (newIndex + 1) + ' of ' + fields.length);

            this.saveFormData();
            this.renderSteps();

            var self = this;
            setTimeout(function () {
                $('.fb-field-item[data-step="' + stepIndex + '"][data-field="' + newIndex + '"]').focus();
            }, 50);
        },

        cancelKeyboardMove: function () {
            if (!this.keyboardGrabbedField || this.originalFieldPosition === null) return;

            var stepIndex = this.keyboardGrabbedField.step;
            var currentIndex = this.keyboardGrabbedField.field;
            var originalIndex = this.originalFieldPosition;

            var fields = this.steps[stepIndex].fields;
            var field = fields[currentIndex];
            fields.splice(currentIndex, 1);
            fields.splice(originalIndex, 0, field);

            this.keyboardGrabbedField = null;
            this.originalFieldPosition = null;

            this.announce('Move cancelled');
            this.saveFormData();
            this.renderSteps();

            var self = this;
            setTimeout(function () {
                $('.fb-field-item[data-step="' + stepIndex + '"][data-field="' + originalIndex + '"]').focus();
            }, 50);
        },

        moveField: function (stepIndex, fieldIndex, direction) {
            var fields = this.steps[stepIndex].fields;
            var newIndex = fieldIndex + direction;

            if (newIndex < 0 || newIndex >= fields.length) return;

            this.saveToHistory('reorder_field');

            var temp = fields[fieldIndex];
            fields[fieldIndex] = fields[newIndex];
            fields[newIndex] = temp;

            this.saveFormData();
            this.renderSteps();

            var fieldLabel = temp.label || 'Untitled';
            this.announce(fieldLabel + ' moved to position ' + (newIndex + 1));

            setTimeout(function () {
                $('.fb-field-item[data-step="' + stepIndex + '"][data-field="' + newIndex + '"]').focus();
            }, 50);
        },

        duplicateField: function (stepIndex, fieldIndex) {

            this.saveToHistory('duplicate_field');

            var field = this.steps[stepIndex].fields[fieldIndex];
            var newField = JSON.parse(JSON.stringify(field));

            if (newField.name) {
                newField.name = newField.name + '_copy_' + Date.now();
            }
            newField.label = (newField.label || 'Untitled') + ' (copy)';

            this.steps[stepIndex].fields.splice(fieldIndex + 1, 0, newField);

            this.saveFormData();
            this.renderSteps();

            this.selectField(stepIndex, fieldIndex + 1);
            this.announce(field.label + ' field duplicated');
        },

        confirmFieldDelete: function (stepIndex, fieldIndex) {
            var field = this.steps[stepIndex].fields[fieldIndex];
            var fieldLabel = field.label || 'Untitled';

            if (this.deleteConfirmPending &&
                this.deleteConfirmPending.step === stepIndex &&
                this.deleteConfirmPending.field === fieldIndex) {

                this.removeField(stepIndex, fieldIndex);
                this.announce(fieldLabel + ' field deleted');
                this.deleteConfirmPending = null;
            } else {

                this.deleteConfirmPending = { step: stepIndex, field: fieldIndex };
                this.announce('Press Delete again to confirm removal of ' + fieldLabel + ' field');

                var self = this;
                setTimeout(function () {
                    if (self.deleteConfirmPending &&
                        self.deleteConfirmPending.step === stepIndex &&
                        self.deleteConfirmPending.field === fieldIndex) {
                        self.deleteConfirmPending = null;
                    }
                }, 3000);
            }
        },

        focusPreviousField: function ($current) {
            var $prev = $current.prev('.fb-field-item');
            if ($prev.length) {
                $prev.focus();
            }
        },

        focusNextField: function ($current) {
            var $next = $current.next('.fb-field-item');
            if ($next.length) {
                $next.focus();
            }
        },

        announce: function (message) {
            var $announcer = $('#fb-announcer');
            if ($announcer.length) {
                $announcer.text(message);

                setTimeout(function () {
                    $announcer.text('');
                }, 1000);
            }
        },

        saveToHistory: function (actionType) {

            if (this.isUndoRedo) return;

            var state = {
                timestamp: Date.now(),
                action: actionType,
                steps: JSON.parse(JSON.stringify(this.steps)),
                activeStep: this.activeStep,
                selectedField: this.selectedField ? { step: this.selectedField.step, field: this.selectedField.field } : null
            };

            if (this.historyIndex < this.history.length - 1) {
                this.history = this.history.slice(0, this.historyIndex + 1);
            }

            this.history.push(state);

            if (this.history.length > this.maxHistory) {
                this.history.shift();
            }

            this.historyIndex = this.history.length - 1;

            this.updateUndoRedoButtons();
        },

        undo: function () {
            if (this.historyIndex <= 0) {
                this.announce('Nothing to undo');
                return;
            }

            this.isUndoRedo = true;

            this.historyIndex--;
            var state = this.history[this.historyIndex];

            this.steps = JSON.parse(JSON.stringify(state.steps));
            this.activeStep = state.activeStep;
            this.selectedField = state.selectedField ? { step: state.selectedField.step, field: state.selectedField.field } : null;

            this.isUndoRedo = false;

            this.renderSteps();

            if (this.selectedField) {
                this.showSettings(this.selectedField.step, this.selectedField.field);
            } else {
                this.closeSettings();
            }

            this.updateUndoRedoButtons();

            var actionLabel = this.getActionLabel(this.history[this.historyIndex + 1].action);
            this.announce('Undo: ' + actionLabel);
        },

        redo: function () {
            if (this.historyIndex >= this.history.length - 1) {
                this.announce('Nothing to redo');
                return;
            }

            this.isUndoRedo = true;

            this.historyIndex++;
            var state = this.history[this.historyIndex];

            this.steps = JSON.parse(JSON.stringify(state.steps));
            this.activeStep = state.activeStep;
            this.selectedField = state.selectedField ? { step: state.selectedField.step, field: state.selectedField.field } : null;

            this.isUndoRedo = false;

            this.renderSteps();

            if (this.selectedField) {
                this.showSettings(this.selectedField.step, this.selectedField.field);
            } else {
                this.closeSettings();
            }

            this.updateUndoRedoButtons();
            this.announce('Redo: ' + this.getActionLabel(state.action));
        },

        getActionLabel: function (action) {
            var labels = {
                'initial': 'initial state',
                'add_field': 'added field',
                'delete_field': 'deleted field',
                'reorder_field': 'reordered field',
                'edit_field': 'edited field',
                'add_step': 'added step',
                'delete_step': 'deleted step',
                'duplicate_field': 'duplicated field'
            };
            return labels[action] || action;
        },

        updateUndoRedoButtons: function () {
            var $undoBtn = $('#btn-undo');
            var $redoBtn = $('#btn-redo');

            if ($undoBtn.length) {
                $undoBtn.prop('disabled', this.historyIndex <= 0);
            }
            if ($redoBtn.length) {
                $redoBtn.prop('disabled', this.historyIndex >= this.history.length - 1);
            }
        },

        canUndo: function () {
            return this.historyIndex > 0;
        },

        canRedo: function () {
            return this.historyIndex < this.history.length - 1;
        },

        addStep: function () {

            this.saveToHistory('add_step');

            var newIndex = this.steps.length + 1;
            this.steps.push({
                id: 'step-' + newIndex,
                title: 'Step ' + newIndex,
                fields: []
            });
            this.activeStep = this.steps.length - 1;
            this.renderSteps();
        },

        removeStep: function (stepIndex) {
            if (this.steps.length <= 1) return;

            this.saveToHistory('delete_step');

            this.steps.splice(stepIndex, 1);
            if (this.activeStep >= this.steps.length) {
                this.activeStep = this.steps.length - 1;
            }
            this.closeSettings();
            this.renderSteps();
        },

        switchStep: function (stepIndex) {
            this.activeStep = stepIndex;
            this.closeSettings();
            this.renderSteps();
        },

        saveFormData: function () {
            var formContent = {};

            this.steps.forEach(function (step, index) {
                var stepId = step.id || 'step-' + (index + 1);
                formContent[stepId] = step.fields;
            });

            if (this.steps.length === 1) {
                formContent = this.steps[0].fields;
            }

            $('#form-content').val(JSON.stringify(formContent));
        },

        saveStepTitles: function () {
            // Use dedicated container for step titles
            var $container = $('#fb-step-titles-container');
            if (!$container.length) {
                // Fallback: create container inside form
                var $form = $('.bb-form-builder-form');
                $form.prepend('<div id="fb-step-titles-container"></div>');
                $container = $('#fb-step-titles-container');
            }

            // Clear existing step title inputs
            $container.empty();

            // Create hidden inputs for step titles and conditions
            this.steps.forEach(function (step) {
                var stepId = step.id || 'step-1';
                var $titleInput = $('<input>')
                    .attr('type', 'hidden')
                    .attr('name', 'properties[step_titles][' + stepId + ']')
                    .val(step.title);
                $container.append($titleInput);

                // Save step conditions
                if (step.condition && step.condition.field) {
                    var $conditionFieldInput = $('<input>')
                        .attr('type', 'hidden')
                        .attr('name', 'properties[step_conditions][' + stepId + '][field]')
                        .val(step.condition.field);
                    var $conditionOperatorInput = $('<input>')
                        .attr('type', 'hidden')
                        .attr('name', 'properties[step_conditions][' + stepId + '][operator]')
                        .val(step.condition.operator || 'checked');
                    var $conditionValueInput = $('<input>')
                        .attr('type', 'hidden')
                        .attr('name', 'properties[step_conditions][' + stepId + '][value]')
                        .val(step.condition.value || '');
                    $container.append($conditionFieldInput);
                    $container.append($conditionOperatorInput);
                    $container.append($conditionValueInput);
                }
            });
        },

        getAllCheckboxAndSelectFields: function () {
            var fields = [];
            this.steps.forEach(function (step, stepIndex) {
                step.fields.forEach(function (field) {
                    if (['checkbox', 'select', 'radio'].includes(field.type) && field.name) {
                        fields.push({
                            name: field.name,
                            label: field.label || field.name,
                            type: field.type,
                            options: field.options || [],
                            stepIndex: stepIndex,
                            stepTitle: step.title
                        });
                    }
                });
            });
            return fields;
        },

        showStepConditionModal: function (stepIndex) {
            var self = this;
            var step = this.steps[stepIndex];
            var condition = step.condition || { field: '', operator: 'checked', value: '' };
            var fields = this.getAllCheckboxAndSelectFields();

            // Only show fields from previous steps
            var availableFields = fields.filter(function (f) {
                return f.stepIndex < stepIndex;
            });

            if (availableFields.length === 0 && stepIndex > 0) {
                alert('No checkbox, radio, or select fields found in previous steps. Add fields to earlier steps first.');
                return;
            }

            if (stepIndex === 0) {
                alert('The first step cannot have conditions since there are no previous steps.');
                return;
            }

            var html = '<div class="fb-step-condition-modal">' +
                '<div class="fb-step-condition-content">' +
                '<div class="fb-step-condition-header">' +
                '<h4>Step Condition: ' + this.escapeHtml(step.title) + '</h4>' +
                '<button type="button" class="fb-step-condition-close">' + icons.close + '</button>' +
                '</div>' +
                '<div class="fb-step-condition-body">' +
                '<p class="fb-step-condition-desc">Show this step only when a specific condition is met. If the condition is not met, the step will be skipped.</p>' +
                '<div class="fb-setting-group">' +
                '<label class="setting-label">Show this step if</label>' +
                '<select class="form-select fb-condition-field" id="step-condition-field">' +
                '<option value="">-- No condition (always show) --</option>';

            availableFields.forEach(function (field) {
                var selected = condition.field === field.name ? ' selected' : '';
                html += '<option value="' + self.escapeHtml(field.name) + '" data-type="' + field.type + '"' + selected + '>' +
                    self.escapeHtml(field.label) + ' (' + field.stepTitle + ')' +
                    '</option>';
            });

            html += '</select></div>' +
                '<div class="fb-setting-group fb-condition-operator-group" style="' + (condition.field ? '' : 'display:none;') + '">' +
                '<label class="setting-label">Condition</label>' +
                '<select class="form-select" id="step-condition-operator">' +
                '<option value="checked"' + (condition.operator === 'checked' ? ' selected' : '') + '>Is checked / selected</option>' +
                '<option value="not_checked"' + (condition.operator === 'not_checked' ? ' selected' : '') + '>Is not checked / not selected</option>' +
                '<option value="equals"' + (condition.operator === 'equals' ? ' selected' : '') + '>Equals value</option>' +
                '<option value="not_equals"' + (condition.operator === 'not_equals' ? ' selected' : '') + '>Does not equal value</option>' +
                '</select></div>' +
                '<div class="fb-setting-group fb-condition-value-group" style="' + (['equals', 'not_equals'].includes(condition.operator) ? '' : 'display:none;') + '">' +
                '<label class="setting-label">Value</label>' +
                '<input type="text" class="form-control" id="step-condition-value" value="' + this.escapeHtml(condition.value || '') + '" placeholder="Enter value to match">' +
                '</div>' +
                '</div>' +
                '<div class="fb-step-condition-footer">' +
                '<button type="button" class="btn btn-secondary fb-step-condition-cancel">Cancel</button>' +
                '<button type="button" class="btn btn-danger fb-step-condition-clear" style="' + (condition.field ? '' : 'display:none;') + '">Remove Condition</button>' +
                '<button type="button" class="btn btn-primary fb-step-condition-save">Save</button>' +
                '</div></div></div>';

            $('body').append(html);

            var $modal = $('.fb-step-condition-modal');
            $modal.data('step-index', stepIndex);

            setTimeout(function () {
                $modal.addClass('show');
            }, 10);
        },

        saveStepCondition: function (stepIndex) {
            var field = $('#step-condition-field').val();
            var operator = $('#step-condition-operator').val();
            var value = $('#step-condition-value').val();

            if (field) {
                this.steps[stepIndex].condition = {
                    field: field,
                    operator: operator,
                    value: value
                };
            } else {
                delete this.steps[stepIndex].condition;
            }

            this.saveStepTitles();
            this.renderSteps();
            this.closeStepConditionModal();
        },

        clearStepCondition: function (stepIndex) {
            delete this.steps[stepIndex].condition;
            this.saveStepTitles();
            this.renderSteps();
            this.closeStepConditionModal();
        },

        closeStepConditionModal: function () {
            var $modal = $('.fb-step-condition-modal');
            $modal.removeClass('show');
            setTimeout(function () {
                $modal.remove();
            }, 300);
        },

        showRenameStepModal: function (stepIndex) {
            var step = this.steps[stepIndex];
            var currentTitle = step.title || 'Step ' + (stepIndex + 1);

            var html = '<div class="fb-step-rename-modal">' +
                '<div class="fb-step-rename-content">' +
                '<div class="fb-step-rename-header">' +
                '<h4>Rename Step</h4>' +
                '<button type="button" class="fb-step-rename-close">' + icons.close + '</button>' +
                '</div>' +
                '<div class="fb-step-rename-body">' +
                '<div class="fb-setting-group">' +
                '<label class="setting-label">Step title</label>' +
                '<input type="text" class="form-control" id="step-rename-input" value="' + this.escapeHtml(currentTitle) + '" placeholder="Enter step title">' +
                '</div>' +
                '</div>' +
                '<div class="fb-step-rename-footer">' +
                '<button type="button" class="btn btn-secondary fb-step-rename-cancel">Cancel</button>' +
                '<button type="button" class="btn btn-primary fb-step-rename-save">Save</button>' +
                '</div></div></div>';

            $('body').append(html);

            var $modal = $('.fb-step-rename-modal');
            $modal.data('step-index', stepIndex);

            setTimeout(function () {
                $modal.addClass('show');
                $('#step-rename-input').focus().select();
            }, 10);
        },

        saveStepRename: function (stepIndex) {
            var newTitle = $('#step-rename-input').val().trim();
            if (newTitle) {
                this.updateStepTitle(stepIndex, newTitle);
            }
            this.closeRenameStepModal();
        },

        closeRenameStepModal: function () {
            var $modal = $('.fb-step-rename-modal');
            $modal.removeClass('show');
            setTimeout(function () {
                $modal.remove();
            }, 300);
        },

        updateStepTitle: function (stepIndex, newTitle) {
            if (stepIndex >= 0 && stepIndex < this.steps.length) {
                this.saveToHistory('update_step_title');
                this.steps[stepIndex].title = newTitle;
                this.saveStepTitles();
                // Update the UI without full re-render
                $('#fb-steps-nav .fb-step-tab[data-step="' + stepIndex + '"] .step-title').text(newTitle);
            }
        },

        escapeHtml: function (text) {
            if (!text) return '';
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        initActions: function () {
            var self = this;

            $(document).on('click', '.add-action-btn[data-action]', function () {
                var actionType = $(this).data('action');
                self.addAction(actionType);
            });

            $(document).on('click', '.remove-action-btn', function (e) {
                e.stopPropagation();
                $(this).closest('.fb-action-card').remove();
                self.updateActionsState();
            });

            $(document).on('click', '.fb-action-card-remove', function (e) {
                e.stopPropagation();
            });
        },

        addAction: function (actionType) {
            var $template = $('#action-template-' + actionType);
            if (!$template.length) return;

            var actionId = Date.now();
            var html = $template.html().replace(/__ACTION_ID__/g, actionId);
            $('#actions-container').append(html);

            this.loadDynamicOptions($('#actions-container').find('.fb-action-card').last());

            this.updateActionsState();

            var $newAction = $('#actions-container').find('.fb-action-card').last();
            if ($newAction.length) {
                $newAction[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
        },

        updateActionsState: function () {
            var count = $('#actions-container').find('.fb-action-card').length;
            $('#actions-count').text(count);

            if (count > 0) {
                $('#actions-empty-state').hide();
            } else {
                $('#actions-empty-state').show();
            }

            // Show/hide database action warning
            var hasDatabaseAction = $('#actions-container').find('.fb-action-card[data-action-type="database"]').length > 0;
            if (hasDatabaseAction) {
                $('#no-database-action-warning').hide();
            } else {
                $('#no-database-action-warning').show();
            }
        },

        loadDynamicOptions: function ($container) {
            var self = this;
            $container = $container || $(document);

            $container.find('.dynamic-options-select').each(function () {
                var $select = $(this);
                var url = $select.data('options-url');
                var selectedValue = $select.data('selected-value') || '';

                if (!url) return;

                $select.html('<option value="">Loading...</option>');

                $.ajax({
                    url: url,
                    method: 'GET',
                    dataType: 'json',
                    success: function (response) {
                        var options = '<option value="">-- Select --</option>';

                        if (response.error) {
                            options = '<option value="">' + (response.message || 'Error loading options') + '</option>';
                        } else if (response.data && response.data.length > 0) {
                            response.data.forEach(function (item) {
                                var selected = item.id === selectedValue ? ' selected' : '';
                                var label = item.name + (item.member_count ? ' (' + item.member_count + ' subscribers)' : '');
                                options += '<option value="' + item.id + '"' + selected + '>' + self.escapeHtml(label) + '</option>';
                            });
                        } else {
                            options = '<option value="">No items found</option>';
                        }

                        $select.html(options);
                    },
                    error: function () {
                        $select.html('<option value="">Failed to load options</option>');
                    }
                });
            });
        },

        bindEvents: function () {
            var self = this;

            $(document).on('click', '.fb-component-card', function () {
                var fieldType = $(this).data('field-type');
                self.addField(self.activeStep, fieldType);

                if (window.innerWidth <= 992) {
                    $('.fb-sidebar').removeClass('show');
                    $('.fb-sidebar-backdrop').removeClass('show').remove();
                }
            });

            $(document).on('click', '.fb-mobile-components-toggle', function () {
                $('.fb-sidebar').addClass('show');
                if (!$('.fb-sidebar-backdrop').length) {
                    $('body').append('<div class="fb-sidebar-backdrop"></div>');
                    setTimeout(function () {
                        $('.fb-sidebar-backdrop').addClass('show');
                    }, 10);
                }
            });

            $(document).on('click', '.fb-sidebar-close, .fb-sidebar-backdrop', function () {
                $('.fb-sidebar').removeClass('show');
                $('.fb-sidebar-backdrop').removeClass('show');
                setTimeout(function () {
                    $('.fb-sidebar-backdrop').remove();
                }, 300);
            });

            $(document).on('click', '.fb-step-tab', function (e) {
                // Skip if clicking on step-remove or step-title (for double-click editing)
                if ($(e.target).hasClass('step-remove') || $(e.target).hasClass('step-title')) {
                    return;
                }
                var stepIndex = $(this).data('step');
                self.switchStep(stepIndex);
            });

            // Single click on step title to switch step (but allow double-click to edit)
            var stepTitleClickTimer = null;
            $(document).on('click', '.fb-step-tab .step-title', function (e) {
                e.stopPropagation();
                var $title = $(this);
                var stepIndex = $title.data('step');

                // Use timeout to differentiate single vs double click
                if (stepTitleClickTimer) {
                    clearTimeout(stepTitleClickTimer);
                    stepTitleClickTimer = null;
                    return; // Double-click will handle this
                }

                stepTitleClickTimer = setTimeout(function () {
                    stepTitleClickTimer = null;
                    self.switchStep(stepIndex);
                }, 250);
            });

            $(document).on('click', '.step-remove', function (e) {
                e.stopPropagation();
                var stepIndex = $(this).data('step');
                self.removeStep(stepIndex);
            });

            $(document).on('click', '.fb-add-condition-btn, .fb-condition-badge-edit, .fb-step-condition-badge', function (e) {
                e.stopPropagation();
                var stepIndex = parseInt($(this).data('step'));
                self.showStepConditionModal(stepIndex);
            });

            // Rename step button
            $(document).on('click', '.fb-rename-step-btn', function (e) {
                e.stopPropagation();
                var stepIndex = parseInt($(this).data('step'));
                self.showRenameStepModal(stepIndex);
            });

            // Rename modal events
            $(document).on('click', '.fb-step-rename-close, .fb-step-rename-cancel', function () {
                self.closeRenameStepModal();
            });

            $(document).on('click', '.fb-step-rename-modal', function (e) {
                if ($(e.target).hasClass('fb-step-rename-modal')) {
                    self.closeRenameStepModal();
                }
            });

            $(document).on('click', '.fb-step-rename-save', function () {
                var stepIndex = $('.fb-step-rename-modal').data('step-index');
                self.saveStepRename(stepIndex);
            });

            $(document).on('keypress', '#step-rename-input', function (e) {
                if (e.which === 13) {
                    var stepIndex = $('.fb-step-rename-modal').data('step-index');
                    self.saveStepRename(stepIndex);
                }
            });

            // Step condition modal events
            $(document).on('click', '.fb-step-condition-close, .fb-step-condition-cancel', function () {
                self.closeStepConditionModal();
            });

            $(document).on('click', '.fb-step-condition-modal', function (e) {
                if ($(e.target).hasClass('fb-step-condition-modal')) {
                    self.closeStepConditionModal();
                }
            });

            $(document).on('click', '.fb-step-condition-save', function () {
                var stepIndex = $('.fb-step-condition-modal').data('step-index');
                self.saveStepCondition(stepIndex);
            });

            $(document).on('click', '.fb-step-condition-clear', function () {
                var stepIndex = $('.fb-step-condition-modal').data('step-index');
                self.clearStepCondition(stepIndex);
            });

            $(document).on('change', '#step-condition-field', function () {
                var hasField = $(this).val() !== '';
                $('.fb-condition-operator-group').toggle(hasField);
                $('.fb-step-condition-clear').toggle(hasField);
                if (!hasField) {
                    $('.fb-condition-value-group').hide();
                }
            });

            $(document).on('change', '#step-condition-operator', function () {
                var showValue = ['equals', 'not_equals'].includes($(this).val());
                $('.fb-condition-value-group').toggle(showValue);
            });

            $(document).on('click', '.add-step-btn', function () {
                self.addStep();
            });

            // Double-click to edit step title
            $(document).on('dblclick', '.fb-step-tab .step-title', function (e) {
                e.stopPropagation();
                var $title = $(this);
                var stepIndex = $title.data('step');
                var currentTitle = self.steps[stepIndex].title;

                // Create inline input
                var $input = $('<input type="text" class="step-title-input">')
                    .val(currentTitle)
                    .css({
                        'width': Math.max($title.width() + 20, 80) + 'px',
                        'padding': '2px 6px',
                        'border': '1px solid var(--fb-primary)',
                        'border-radius': '4px',
                        'font-size': 'inherit',
                        'font-weight': 'inherit',
                        'outline': 'none',
                        'background': 'var(--fb-background)',
                        'color': 'inherit'
                    });

                $title.hide().after($input);
                $input.focus().select();

                var finishEditing = function () {
                    var newTitle = $input.val().trim() || currentTitle;
                    $input.remove();
                    $title.show();
                    if (newTitle !== currentTitle) {
                        self.updateStepTitle(stepIndex, newTitle);
                    }
                };

                $input.on('blur', finishEditing);
                $input.on('keydown', function (e) {
                    if (e.key === 'Enter') {
                        e.preventDefault();
                        finishEditing();
                    } else if (e.key === 'Escape') {
                        $input.val(currentTitle);
                        finishEditing();
                    }
                });
            });

            $(document).on('click', '.fb-field-item', function (e) {

                if ($(e.target).closest('.fb-editable, .btn-field-action').length) {
                    return;
                }
                var stepIndex = $(this).data('step');
                var fieldIndex = $(this).data('field');
                self.selectField(stepIndex, fieldIndex);
            });

            $(document).on('click', '.btn-settings', function () {
                var stepIndex = $(this).data('step');
                var fieldIndex = $(this).data('field');
                self.selectField(stepIndex, fieldIndex);
            });

            $(document).on('click', '.btn-delete', function () {
                var stepIndex = $(this).data('step');
                var fieldIndex = $(this).data('field');
                self.removeField(stepIndex, fieldIndex);
            });

            $(document).on('click', '.fb-settings-close', function () {
                self.closeSettings();
            });

            $(document).on('change input', '.field-setting', function () {
                var prop = $(this).data('prop');
                var value = $(this).attr('type') === 'checkbox' ? $(this).is(':checked') : $(this).val();
                self.updateField(prop, value);
            });

            $(document).on('click', '.width-option', function () {
                var width = $(this).data('width');
                $(this).siblings().removeClass('active');
                $(this).addClass('active');
                self.updateField('width', width.toString());
            });

            $(document).on('change', '.field-setting-checkbox', function () {
                var $container = $(this).closest('.fb-default-checkboxes');
                var values = [];
                $container.find('.field-setting-checkbox:checked').each(function () {
                    values.push($(this).val());
                });
                self.updateField('default_value', values);
            });

            $(document).on('input', '.fb-form-name-input', function () {
                $('.fb-canvas-title').text($(this).val() || 'Untitled Form');
            });

            $('form.bb-form-builder-form').on('submit', function () {
                self.saveFormData();
            });

            $(document).on('click', '#btn-preview-form', function () {
                self.showPreview();
            });

            $(document).on('click', '.fb-preview-device-switcher button', function () {
                var device = $(this).data('device');
                $('.fb-preview-device-switcher button').removeClass('active');
                $(this).addClass('active');
                $('.fb-preview-frame').attr('data-device', device);
            });

            $(document).on('click', '.fb-editable', function (e) {
                e.stopPropagation();

                var $editable = $(this);

                if ($editable.hasClass('is-editing')) return;

                self.exitInlineEdit();

                $editable.attr('contenteditable', 'true')
                    .addClass('is-editing')
                    .focus();

                var range = document.createRange();
                range.selectNodeContents($editable[0]);
                var sel = window.getSelection();
                sel.removeAllRanges();
                sel.addRange(range);
            });

            $(document).on('keydown', '.fb-editable.is-editing', function (e) {
                var $editable = $(this);

                if (e.key === 'Enter') {
                    e.preventDefault();
                    self.saveInlineEdit($editable);
                    $editable.blur();
                } else if (e.key === 'Escape') {
                    e.preventDefault();
                    self.cancelInlineEdit($editable);
                } else if (e.key === 'Tab') {
                    e.preventDefault();
                    self.saveInlineEdit($editable);

                    var $allEditables = $('.fb-editable');
                    var currentIndex = $allEditables.index($editable);
                    var nextIndex = e.shiftKey ? currentIndex - 1 : currentIndex + 1;

                    if (nextIndex >= 0 && nextIndex < $allEditables.length) {
                        $allEditables.eq(nextIndex).trigger('click');
                    } else {
                        $editable.blur();
                    }
                }
            });

            $(document).on('blur', '.fb-editable.is-editing', function () {
                var $editable = $(this);
                self.saveInlineEdit($editable);
            });

            $(document).on('mousedown', '.fb-editable.is-editing', function (e) {
                e.stopPropagation();
            });

            $(document).on('keydown', '.fb-field-item', function (e) {
                var $field = $(this);
                var stepIndex = parseInt($field.data('step'));
                var fieldIndex = parseInt($field.data('field'));

                if ($field.find('.fb-editable.is-editing').length) {
                    return;
                }

                switch (e.key) {
                    case ' ':
                    case 'Enter':

                        if (!$(e.target).closest('.fb-editable').length) {
                            e.preventDefault();
                            self.toggleKeyboardGrab(stepIndex, fieldIndex);
                        }
                        break;

                    case 'ArrowUp':
                        e.preventDefault();
                        if (self.keyboardGrabbedField) {
                            self.moveFieldKeyboard(stepIndex, fieldIndex, -1);
                        } else {
                            self.focusPreviousField($field);
                        }
                        break;

                    case 'ArrowDown':
                        e.preventDefault();
                        if (self.keyboardGrabbedField) {
                            self.moveFieldKeyboard(stepIndex, fieldIndex, 1);
                        } else {
                            self.focusNextField($field);
                        }
                        break;

                    case 'Escape':
                        if (self.keyboardGrabbedField) {
                            e.preventDefault();
                            self.cancelKeyboardMove();
                        }
                        break;

                    case 'Delete':
                    case 'Backspace':
                        if (!$(e.target).closest('.fb-editable').length) {
                            e.preventDefault();
                            self.confirmFieldDelete(stepIndex, fieldIndex);
                        }
                        break;

                    case 'd':
                    case 'D':
                        if (!$(e.target).closest('.fb-editable').length && !e.ctrlKey && !e.metaKey) {
                            e.preventDefault();
                            self.duplicateField(stepIndex, fieldIndex);
                        }
                        break;
                }
            });

            $(document).on('blur', '.fb-field-item', function () {
                self.deleteConfirmPending = null;
            });

            $(document).on('click', '.btn-move-up', function (e) {
                e.stopPropagation();
                var stepIndex = parseInt($(this).data('step'));
                var fieldIndex = parseInt($(this).data('field'));
                self.moveField(stepIndex, fieldIndex, -1);
            });

            $(document).on('click', '.btn-move-down', function (e) {
                e.stopPropagation();
                var stepIndex = parseInt($(this).data('step'));
                var fieldIndex = parseInt($(this).data('field'));
                self.moveField(stepIndex, fieldIndex, 1);
            });

            $(document).on('click', '.btn-duplicate', function (e) {
                e.stopPropagation();
                var stepIndex = parseInt($(this).data('step'));
                var fieldIndex = parseInt($(this).data('field'));
                self.duplicateField(stepIndex, fieldIndex);
            });

            $(document).on('click', '#btn-undo', function () {
                self.undo();
            });

            $(document).on('click', '#btn-redo', function () {
                self.redo();
            });

            $(document).on('keydown', function (e) {

                var tagName = e.target.tagName.toLowerCase();
                if (tagName === 'input' || tagName === 'textarea' || e.target.isContentEditable) {
                    return;
                }

                if ((e.ctrlKey || e.metaKey) && e.key === 'z' && !e.shiftKey) {
                    e.preventDefault();
                    self.undo();
                }

                if (((e.ctrlKey || e.metaKey) && e.shiftKey && e.key === 'z') ||
                    ((e.ctrlKey || e.metaKey) && e.key === 'y')) {
                    e.preventDefault();
                    self.redo();
                }
            });
        },

        showPreview: function () {
            var html = this.generatePreviewHtml();
            $('#fb-preview-content').html(html);
            var modal = new bootstrap.Modal(document.getElementById('form-preview-modal'));
            modal.show();
        },

        generatePreviewHtml: function () {
            var self = this;
            var hasMultipleSteps = this.steps.length > 1;
            var wizardLayout = $('select[name="properties[layout]"]').val() || 'horizontal';
            var formSize = $('select[name="properties[form_size]"]').val() || 'lg';
            var formAlignment = $('select[name="properties[form_alignment]"]').val() || 'center';
            var formStyle = $('select[name="properties[form_style]"]').val() || 'default';
            var formTitle = $('input[name="properties[form_title]"]').val() || '';
            var formDescription = $('textarea[name="properties[form_description]"]').val() || '';
            var formMargin = $('input[name="properties[form_margin]"]').val() || '';
            var formPadding = $('input[name="properties[form_padding]"]').val() || '';

            var containerClasses = 'fb-form-container fb-form-size-' + formSize + ' fb-form-alignment-' + formAlignment;
            if (formStyle && formStyle !== 'default') {
                containerClasses += ' fb-form-style-' + formStyle;
            }
            if (hasMultipleSteps && wizardLayout === 'vertical') {
                containerClasses += ' fb-wizard-vertical';
            }

            // Build inline styles for margin and padding
            var containerStyles = [];
            if (formMargin) {
                containerStyles.push('margin: ' + formMargin);
            }
            if (formPadding) {
                containerStyles.push('--fb-form-padding: ' + formPadding);
            }
            var styleAttr = containerStyles.length ? ' style="' + containerStyles.join('; ') + '"' : '';

            var html = '<div class="' + containerClasses + '"' + styleAttr + '>';

            // Add form header with title and description
            if (formTitle || formDescription) {
                html += '<div class="fb-form-header">';
                if (formTitle) {
                    html += '<h2 class="fb-form-title">' + this.escapeHtml(formTitle) + '</h2>';
                }
                if (formDescription) {
                    html += '<p class="fb-form-description">' + this.escapeHtml(formDescription) + '</p>';
                }
                html += '</div>';
            }

            html += '<form class="fb-form">';
            html += '<div class="fb-form-message"></div>';

            if (hasMultipleSteps) {
                html += this.generateWizardPreview(wizardLayout);
            } else {
                html += '<div class="row">';
                if (this.steps[0] && this.steps[0].fields) {
                    this.steps[0].fields.forEach(function (field) {
                        html += self.generateFieldPreviewHtml(field);
                    });
                }
                html += '</div>';
                html += '<div class="fb-form-submit"><button type="button" class="fb-form-submit-btn">Submit</button></div>';
            }

            html += '</form></div>';
            return html;
        },

        generateWizardPreview: function (layout) {
            var self = this;
            var html = '';
            var stepCount = this.steps.length;

            html += '<div class="fb-form-wizard fb-wizard-' + layout + '" data-steps="' + stepCount + '">';

            if (layout === 'vertical') {
                html += '<div class="fb-wizard-vertical-container">';
                html += '<div class="fb-wizard-sidebar"><ul class="fb-wizard-steps">';
                this.steps.forEach(function (step, index) {
                    html += '<li class="fb-wizard-step' + (index === 0 ? ' active' : '') + '" data-step="' + index + '">';
                    html += '<div class="step-indicator-wrapper"><span class="step-indicator"></span>';
                    if (index < stepCount - 1) {
                        html += '<span class="step-connector"></span>';
                    }
                    html += '</div>';
                    html += '<div class="step-info"><span class="step-title">' + self.formatStepTitle(step.title) + '</span>';
                    html += '<span class="step-subtitle">Step ' + (index + 1) + ' of ' + stepCount + '</span></div>';
                    html += '</li>';
                });
                html += '</ul></div>';

                html += '<div class="fb-wizard-main"><div class="fb-wizard-content">';
                this.steps.forEach(function (step, index) {
                    html += '<div class="fb-wizard-panel' + (index === 0 ? ' active' : '') + '" data-step="' + index + '">';
                    html += '<div class="fb-wizard-panel-header"><h3 class="fb-wizard-panel-title">' + self.formatStepTitle(step.title) + '</h3></div>';
                    html += '<div class="row">';
                    if (step.fields) {
                        step.fields.forEach(function (field) {
                            html += self.generateFieldPreviewHtml(field);
                        });
                    }
                    html += '</div></div>';
                });
                html += '</div>';
                html += '<div class="fb-wizard-nav"><div class="fb-wizard-nav-left"><button type="button" class="fb-wizard-prev" style="display:none;">Previous</button></div>';
                html += '<div class="fb-wizard-nav-right"><button type="button" class="fb-wizard-next">Next</button>';
                html += '<button type="button" class="fb-wizard-submit" style="display:none;">Submit</button></div></div>';
                html += '</div></div>';
            } else {
                html += '<ul class="fb-wizard-steps">';
                this.steps.forEach(function (step, index) {
                    html += '<li class="fb-wizard-step' + (index === 0 ? ' active' : '') + '" data-step="' + index + '">';
                    html += '<span class="step-indicator"></span>';
                    html += '<span class="step-title">' + self.formatStepTitle(step.title) + '</span>';
                    html += '</li>';
                });
                html += '</ul>';

                html += '<div class="fb-wizard-content">';
                this.steps.forEach(function (step, index) {
                    html += '<div class="fb-wizard-panel' + (index === 0 ? ' active' : '') + '" data-step="' + index + '">';
                    html += '<div class="row">';
                    if (step.fields) {
                        step.fields.forEach(function (field) {
                            html += self.generateFieldPreviewHtml(field);
                        });
                    }
                    html += '</div></div>';
                });
                html += '</div>';
                html += '<div class="fb-wizard-nav"><div class="fb-wizard-nav-left"><button type="button" class="fb-wizard-prev" style="display:none;">Previous</button></div>';
                html += '<div class="fb-wizard-nav-right"><button type="button" class="fb-wizard-next">Next</button>';
                html += '<button type="button" class="fb-wizard-submit" style="display:none;">Submit</button></div></div>';
            }

            html += '</div>';
            return html;
        },

        formatStepTitle: function (title) {
            return title.replace(/[-_]/g, ' ').replace(/\b\w/g, function (l) { return l.toUpperCase(); });
        },

        generateFieldPreviewHtml: function (field) {
            var width = field.width || '12';
            var fieldName = field.name || 'field';
            var fieldLabel = field.label || field.name || 'Field';
            var hideLabel = field.hide_label || false;
            var showLabel = !hideLabel && fieldLabel.trim() !== '';
            var html = '';

            if (field.type === 'hidden') {
                html += '<input type="hidden" name="' + fieldName + '">';
                return html;
            }

            if (field.type === 'html') {
                html += '<div class="col-md-' + width + ' mb-3 fb-form-field fb-form-field-html">';
                html += '<div class="fb-html-content">' + (field.content || field.html_content || '') + '</div>';
                html += '</div>';
                return html;
            }

            html += '<div class="col-md-' + width + ' mb-3 fb-form-field fb-form-field-' + field.type + '">';

            if (!['checkbox', 'radio'].includes(field.type) && showLabel) {
                html += '<label class="form-label' + (field.required ? ' required' : '') + '">' + fieldLabel + '</label>';
            }

            switch (field.type) {
                case 'text':
                case 'email':
                case 'number':
                case 'date':
                case 'time':
                    var inputType = field.type;
                    if (field.type === 'datetime') inputType = 'datetime-local';
                    html += '<input type="' + inputType + '" name="' + fieldName + '" class="form-control" placeholder="' + (field.placeholder || '') + '"' + (field.required ? ' required' : '') + '>';
                    html += '<div class="invalid-feedback"></div>';
                    break;
                case 'phone':
                    html += '<input type="tel" name="' + fieldName + '" class="form-control" placeholder="' + (field.placeholder || '') + '"' + (field.required ? ' required' : '') + '>';
                    html += '<div class="invalid-feedback"></div>';
                    break;
                case 'textarea':
                    html += '<textarea name="' + fieldName + '" class="form-control" rows="' + (field.rows || 4) + '" placeholder="' + (field.placeholder || '') + '"' + (field.required ? ' required' : '') + '></textarea>';
                    html += '<div class="invalid-feedback"></div>';
                    break;
                case 'select':
                case 'autocomplete':
                    html += '<select name="' + fieldName + '" class="form-select"' + (field.required ? ' required' : '') + '>';
                    html += '<option value="">' + (field.placeholder || 'Select an option') + '</option>';
                    if (field.options) {
                        field.options.forEach(function (opt) {
                            var optValue = typeof opt === 'object' ? (opt.value || '') : opt;
                            var optLabel = typeof opt === 'object' ? (opt.label || optValue) : opt;
                            html += '<option value="' + optValue + '">' + optLabel + '</option>';
                        });
                    }
                    html += '</select>';
                    html += '<div class="invalid-feedback"></div>';
                    break;
                case 'radio':
                    var showRequiredOnRadioOption = field.required && !showLabel;
                    if (showLabel) {
                        html += '<label class="form-label' + (field.required ? ' required' : '') + '">' + fieldLabel + '</label>';
                    }
                    if (field.options) {
                        field.options.forEach(function (opt, i) {
                            var optValue = typeof opt === 'object' ? (opt.value || '') : opt;
                            var optLabel = typeof opt === 'object' ? (opt.label || optValue) : opt;
                            html += '<div class="form-check">';
                            html += '<input type="radio" name="' + fieldName + '" value="' + optValue + '" class="form-check-input" id="preview-' + fieldName + '-' + i + '"' + (field.required && i === 0 ? ' required' : '') + '>';
                            html += '<label class="form-check-label' + (showRequiredOnRadioOption && i === 0 ? ' required' : '') + '" for="preview-' + fieldName + '-' + i + '">' + optLabel + '</label>';
                            html += '</div>';
                        });
                    }
                    html += '<div class="invalid-feedback"></div>';
                    break;
                case 'checkbox':
                    var showRequiredOnCheckboxOption = field.required && !showLabel;
                    if (showLabel) {
                        html += '<label class="form-label' + (field.required ? ' required' : '') + '">' + fieldLabel + '</label>';
                    }
                    if (field.options && field.options.length > 0) {
                        field.options.forEach(function (opt, i) {
                            var optValue = typeof opt === 'object' ? (opt.value || '') : opt;
                            var optLabel = typeof opt === 'object' ? (opt.label || optValue) : opt;
                            html += '<div class="form-check">';
                            html += '<input type="checkbox" name="' + fieldName + '[]" value="' + optValue + '" class="form-check-input" id="preview-' + fieldName + '-' + i + '">';
                            html += '<label class="form-check-label' + (showRequiredOnCheckboxOption && i === 0 ? ' required' : '') + '" for="preview-' + fieldName + '-' + i + '">' + optLabel + '</label>';
                            html += '</div>';
                        });
                    } else {
                        html += '<div class="form-check">';
                        html += '<input type="checkbox" name="' + fieldName + '" value="1" class="form-check-input" id="preview-' + fieldName + '">';
                        html += '<label class="form-check-label' + (showRequiredOnCheckboxOption ? ' required' : '') + '" for="preview-' + fieldName + '">' + fieldLabel + '</label>';
                        html += '</div>';
                    }
                    html += '<div class="invalid-feedback"></div>';
                    break;
                case 'file':
                case 'image':
                    var isImage = field.type === 'image';
                    html += '<div class="fb-file-upload-wrapper" data-type="' + field.type + '">';
                    html += '<div class="fb-file-dropzone">';
                    html += '<input type="file" name="' + fieldName + '" class="fb-file-input"' + (field.required ? ' required' : '') + (isImage ? ' accept="image/*"' : '') + '>';
                    html += '<div class="fb-file-dropzone-content">';
                    if (isImage) {
                        html += '<svg class="fb-file-icon" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg>';
                    } else {
                        html += '<svg class="fb-file-icon" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="17 8 12 3 7 8"/><line x1="12" y1="3" x2="12" y2="15"/></svg>';
                    }
                    html += '<span class="fb-file-text">' + (isImage ? 'Choose an image' : 'Choose a file') + '</span>';
                    html += '</div>';
                    html += '<button type="button" class="fb-file-browse-btn">Browse</button>';
                    html += '</div>';
                    html += '<div class="fb-file-list"></div>';
                    html += '<div class="invalid-feedback"></div>';
                    html += '</div>';
                    break;
                default:
                    html += '<input type="text" name="' + fieldName + '" class="form-control" placeholder="' + (field.placeholder || '') + '"' + (field.required ? ' required' : '') + '>';
                    html += '<div class="invalid-feedback"></div>';
            }

            html += '</div>';
            return html;
        }
    };

    $(document).ready(function () {
        FbFormBuilder.init();
    });

})(jQuery);
