(function (window, document) {
    'use strict';

    var BbFormBuilder = {
        version: '1.2.0',
        baseUrl: null,
        loadedAssets: {
            css: [],
            js: []
        },
        instances: [],

        embed: function (options) {
            var self = this;

            if (!options || !options.container || !options.formCode) {
                console.error('BbFormBuilder: container and formCode are required');
                return;
            }

            var container = document.querySelector(options.container);
            if (!container) {
                console.error('BbFormBuilder: Container element not found:', options.container);
                return;
            }

            if (!this.baseUrl) {
                this.baseUrl = options.baseUrl || this.detectBaseUrl();
            }

            // Version compatibility check
            if (options.version && options.version !== this.version) {
                console.warn('BbFormBuilder: Version mismatch. Script version: ' + this.version + ', Expected version: ' + options.version + '. Please update your embed code.');
            }

            container.innerHTML = '<div class="fb-embed-loading" style="text-align: center; padding: 40px; color: #6b7280;"><svg width="40" height="40" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><style>.spinner{transform-origin:center;animation:spin 1s linear infinite}@keyframes spin{100%{transform:rotate(360deg)}}</style><circle class="spinner" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2" fill="none" stroke-dasharray="31.4 31.4"/></svg><p style="margin-top: 12px;">Loading form...</p></div>';

            this.fetchForm(options.formCode, function (error, data) {
                if (error) {
                    container.innerHTML = '<div class="fb-embed-error" style="text-align: center; padding: 40px; color: #ef4444;"><p>' + self.escapeHtml(error) + '</p></div>';
                    if (options.onError) options.onError(error);
                    return;
                }

                self.loadAssets(options.formCode, function () {
                    container.innerHTML = data.html;

                    var apiSubmitUrl = self.baseUrl + '/api/bb-form-builder/form/' + encodeURIComponent(data.form_hash) + '/submit';

                    var instance = {
                        container: container,
                        formCode: options.formCode,
                        formHash: data.form_hash,
                        submitUrl: apiSubmitUrl, // Use API endpoint instead of regular submit
                        csrfToken: data.csrf_token,
                        options: options
                    };
                    self.instances.push(instance);

                    self.initForm(instance);

                    if (options.onLoad) {
                        options.onLoad(instance);
                    }
                });
            });
        },

        detectBaseUrl: function () {
            var scripts = document.getElementsByTagName('script');
            for (var i = 0; i < scripts.length; i++) {
                var src = scripts[i].src;
                if (src && src.indexOf('bb-form-builder') !== -1 && src.indexOf('embed.js') !== -1) {
                    var match = src.match(/^(https?:\/\/[^\/]+)/);
                    if (match) {
                        return match[1];
                    }
                }
            }
            return window.location.origin;
        },

        fetchForm: function (formCode, callback) {
            var url = this.baseUrl + '/api/bb-form-builder/form/' + encodeURIComponent(formCode);

            this.ajax({
                url: url,
                method: 'GET',
                success: function (response) {
                    if (response.error) {
                        callback(response.message || 'Failed to load form');
                    } else {
                        callback(null, response.data);
                    }
                },
                error: function (xhr, status, error) {
                    var message = 'Failed to load form';
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.message) message = response.message;
                    } catch (e) {}
                    callback(message);
                }
            });
        },

        loadAssets: function (formCode, callback) {
            var self = this;
            var url = this.baseUrl + '/api/bb-form-builder/form/' + encodeURIComponent(formCode) + '/assets';

            this.ajax({
                url: url,
                method: 'GET',
                success: function (response) {
                    if (response.error || !response.data) {
                        callback();
                        return;
                    }

                    var cssAssets = response.data.css || [];
                    var jsAssets = response.data.js || [];

                    cssAssets.forEach(function (url) {
                        self.loadCSS(url);
                    });

                    self.loadScriptsSequentially(jsAssets, callback);
                },
                error: function () {
                    callback();
                }
            });
        },

        loadCSS: function (url) {
            if (this.loadedAssets.css.indexOf(url) !== -1) return;

            var link = document.createElement('link');
            link.rel = 'stylesheet';
            link.type = 'text/css';
            link.href = url;
            document.head.appendChild(link);
            this.loadedAssets.css.push(url);
        },

        loadScriptsSequentially: function (urls, callback, index) {
            index = index || 0;

            if (index >= urls.length) {
                callback();
                return;
            }

            var self = this;
            var url = urls[index];

            if (this.loadedAssets.js.indexOf(url) !== -1) {
                this.loadScriptsSequentially(urls, callback, index + 1);
                return;
            }

            var script = document.createElement('script');
            script.type = 'text/javascript';
            script.src = url;
            script.onload = function () {
                self.loadedAssets.js.push(url);
                self.loadScriptsSequentially(urls, callback, index + 1);
            };
            script.onerror = function () {
                console.warn('BbFormBuilder: Failed to load script:', url);
                self.loadScriptsSequentially(urls, callback, index + 1);
            };
            document.body.appendChild(script);
        },

        initForm: function (instance) {
            var self = this;
            var form = instance.container.querySelector('form.fb-form');

            if (!form) return;

            var csrfInput = form.querySelector('input[name="_token"]');
            if (!csrfInput) {
                csrfInput = document.createElement('input');
                csrfInput.type = 'hidden';
                csrfInput.name = '_token';
                csrfInput.value = instance.csrfToken;
                form.appendChild(csrfInput);
            } else {
                csrfInput.value = instance.csrfToken;
            }

            form.action = instance.submitUrl;

            form.addEventListener('submit', function (e) {
                e.preventDefault();
                self.submitForm(instance, form);
            });

            if (window.FbFormPublic && typeof window.FbFormPublic.init === 'function') {
                window.FbFormPublic.init();
            }

            self.initPhoneFields(instance.container);
        },

        initPhoneFields: function (container) {
            if (typeof window.intlTelInput !== 'function') {
                return;
            }

            var phoneInputs = container.querySelectorAll('.js-phone-number-mask[data-country-code-selection="true"]');
            phoneInputs.forEach(function (element) {
                if (element.dataset.itiInitialized === 'true') {
                    return;
                }

                var config = {
                    initialCountry: 'auto',
                    geoIpLookup: function (callback) {
                        fetch('https://ipinfo.io/json', { credentials: 'omit' })
                            .then(function (response) { return response.json(); })
                            .then(function (data) {
                                callback(data && data.country ? data.country.toLowerCase() : 'us');
                            })
                            .catch(function () {
                                callback('us');
                            });
                    },
                    separateDialCode: true,
                    nationalMode: false,
                    autoHideDialCode: false
                };

                var iti = window.intlTelInput(element, config);
                element.dataset.itiInitialized = 'true';

                var hiddenFieldId = element.id + '-full';
                var hiddenField = document.getElementById(hiddenFieldId);

                if (hiddenField) {
                    var updateHiddenField = function () {
                        var fullNumber = iti.getNumber();
                        if (fullNumber) {
                            hiddenField.value = fullNumber;
                        } else if (element.value) {
                            var selectedCountryData = iti.getSelectedCountryData();
                            if (selectedCountryData && selectedCountryData.dialCode) {
                                hiddenField.value = '+' + selectedCountryData.dialCode + element.value.replace(/\D/g, '');
                            } else {
                                hiddenField.value = element.value;
                            }
                        }
                    };

                    element.addEventListener('countrychange', updateHiddenField);
                    element.addEventListener('input', updateHiddenField);
                    element.addEventListener('blur', updateHiddenField);

                    var form = element.closest('form');
                    if (form) {
                        form.addEventListener('submit', function () {
                            updateHiddenField();
                        });
                    }
                }
            });
        },

        submitForm: function (instance, form) {
            var self = this;
            var options = instance.options;

            if (options.onSubmit) {
                var shouldContinue = options.onSubmit(form);
                if (shouldContinue === false) return;
            }

            var submitBtn = form.querySelector('button[type="submit"], .fb-form-submit-btn, .fb-wizard-submit');
            var originalBtnText = submitBtn ? submitBtn.innerHTML : '';
            if (submitBtn) {
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<svg width="16" height="16" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" style="animation: spin 1s linear infinite;"><style>@keyframes spin{100%{transform:rotate(360deg)}}</style><circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="2" fill="none" stroke-dasharray="31.4 31.4"/></svg> Submitting...';
            }

            var formData = new FormData(form);

            this.ajax({
                url: instance.submitUrl,
                method: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function (response) {
                    if (submitBtn) {
                        submitBtn.disabled = false;
                        submitBtn.innerHTML = originalBtnText;
                    }

                    if (response.error) {
                        self.showMessage(instance.container, 'error', response.message || 'Submission failed');
                        if (options.onError) options.onError(response.message, response);
                    } else {
                        self.showMessage(instance.container, 'success', response.message || 'Form submitted successfully!');
                        form.reset();
                        if (options.onSuccess) options.onSuccess(response);

                        if (response.data && response.data.redirect_url) {
                            setTimeout(function () {
                                window.location.href = response.data.redirect_url;
                            }, 1500);
                        }
                    }
                },
                error: function (xhr, status, error) {
                    if (submitBtn) {
                        submitBtn.disabled = false;
                        submitBtn.innerHTML = originalBtnText;
                    }

                    var message = 'Submission failed. Please try again.';
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.message) message = response.message;
                        if (response.errors) {
                            var errors = [];
                            for (var field in response.errors) {
                                errors.push(response.errors[field][0]);
                            }
                            if (errors.length) message = errors.join('<br>');
                        }
                    } catch (e) {}

                    self.showMessage(instance.container, 'error', message);
                    if (options.onError) options.onError(message);
                }
            });
        },

        showMessage: function (container, type, message) {
            var messageEl = container.querySelector('.fb-form-message');
            if (!messageEl) {
                messageEl = document.createElement('div');
                messageEl.className = 'fb-form-message';
                var form = container.querySelector('form.fb-form');
                if (form) {
                    form.insertBefore(messageEl, form.firstChild);
                } else {
                    container.insertBefore(messageEl, container.firstChild);
                }
            }

            messageEl.className = 'fb-form-message fb-form-message-' + type + ' show';
            messageEl.innerHTML = '<div class="fb-form-message-content">' + message + '</div>';

            if (type === 'success') {
                setTimeout(function () {
                    messageEl.classList.remove('show');
                }, 5000);
            }
        },

        ajax: function (options) {
            var xhr = new XMLHttpRequest();
            xhr.open(options.method || 'GET', options.url, true);

            if (options.contentType !== false) {
                xhr.setRequestHeader('Content-Type', options.contentType || 'application/json');
            }
            xhr.setRequestHeader('Accept', 'application/json');
            xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

            xhr.onreadystatechange = function () {
                if (xhr.readyState === 4) {
                    if (xhr.status >= 200 && xhr.status < 300) {
                        try {
                            var response = JSON.parse(xhr.responseText);
                            if (options.success) options.success(response);
                        } catch (e) {
                            if (options.error) options.error(xhr, 'parseerror', e);
                        }
                    } else {
                        if (options.error) options.error(xhr, xhr.statusText, null);
                    }
                }
            };

            if (options.data) {
                if (options.processData === false) {
                    xhr.send(options.data);
                } else {
                    xhr.send(JSON.stringify(options.data));
                }
            } else {
                xhr.send();
            }
        },

        escapeHtml: function (text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        destroy: function (container) {
            var selector = typeof container === 'string' ? container : null;
            var element = selector ? document.querySelector(selector) : container;

            if (element) {
                element.innerHTML = '';
                this.instances = this.instances.filter(function (instance) {
                    return instance.container !== element;
                });
            }
        }
    };

    window.BbFormBuilder = BbFormBuilder;

})(window, document);
